package abc

import (
	"fmt"
	"net/http"
	"strconv"

	"github.com/go-resty/resty/v2"
)

const URL = "https://abc-back.yandex-team.ru/api/v4/"

func uniquifyLogins(logins []string) []string {
	var uniqueLogins []string
	loginsMap := make(map[string]interface{})
	for _, login := range logins {
		if _, ok := loginsMap[login]; !ok && login != "" {
			uniqueLogins = append(uniqueLogins, login)
			loginsMap[login] = nil
		}
	}

	return uniqueLogins
}

type Client struct {
	c *resty.Client
}

func NewClient(token string) *Client {
	return &Client{
		c: resty.New().
			SetBaseURL(URL).
			SetAuthScheme("OAuth").
			SetAuthToken(token).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
	}
}

type Person struct {
	Login string `json:"login,omitempty"`
}

type ServicesResponseResult struct {
	ID int `json:"id"`
}

type ServicesResponse struct {
	Results []ServicesResponseResult `json:"results"`
}

type ServiceMembersResult struct {
	Person Person `json:"person"`
}

type GetServiceMembersResponse struct {
	Results []ServiceMembersResult `json:"results"`
}

func (c *Client) getServices(queryParams map[string]string) (*ServicesResponse, error) {
	response := &ServicesResponse{}
	resp, err := c.c.R().SetQueryParams(queryParams).SetResult(response).Get("/services/")
	if err != nil {
		return nil, err
	}
	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("invalid status code: %d", resp.StatusCode())
	}

	return response, nil
}

func (c *Client) GetServiceIDBySlug(slug string) (int, error) {
	resp, err := c.getServices(map[string]string{"slug": slug})
	if err != nil {
		return 0, fmt.Errorf("get ABC services request failed: %w", err)
	}
	count := len(resp.Results)
	if count == 0 {
		return 0, fmt.Errorf("ABC service with slug %q not found", slug)
	}
	if count != 1 {
		return 0, fmt.Errorf("expected 1 ABC service with slug %q, got %d", slug, count)
	}

	return resp.Results[0].ID, nil
}

func (c *Client) getServiceMembersLogins(params map[string]string) ([]string, error) {
	params["fields"] = "person.login"

	var members GetServiceMembersResponse
	resp, err := c.c.R().SetQueryParams(params).SetResult(&members).Get("/services/members/")
	if err != nil {
		return nil, fmt.Errorf("request to ABC failed: %w", err)
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	logins := make([]string, 0)
	for _, result := range members.Results {
		logins = append(logins, result.Person.Login)
	}

	return uniquifyLogins(logins), nil
}

func (c *Client) GetServiceMembers(abcServiceID int) ([]string, error) {
	params := map[string]string{
		"service": strconv.Itoa(abcServiceID),
	}
	return c.getServiceMembersLogins(params)
}

func (c *Client) IsUserServiceMember(slug string, username string) (bool, error) {
	params := map[string]string{
		"service__slug": slug,
		"person":        username,
	}
	members, err := c.getServiceMembersLogins(params)
	if err != nil {
		return false, err
	}
	for _, m := range members {
		if m == username {
			return true, nil
		}
	}
	return false, nil
}

func (c *Client) GetServiceMembersBySlug(abcServiceSlug string) ([]string, error) {
	params := map[string]string{
		"service__slug": abcServiceSlug,
	}
	return c.getServiceMembersLogins(params)
}
