package idm

import (
	"fmt"

	"github.com/go-resty/resty/v2"
	"github.com/gofrs/uuid"
)

const URL = "https://idm-api.yandex-team.ru/api/v1/"

type Client struct {
	c *resty.Client
}

func NewClient(opts ...option) *Client {
	c := resty.New().
		SetBaseURL(URL).
		SetAllowGetMethodPayload(true).
		SetHeader("Content-Type", "application/json").
		SetHeader("Accept", "application/json")

	for _, opt := range opts {
		c = opt.configure(c)
	}

	return &Client{
		c: c,
	}
}

func (c *Client) doRequest(url string, method string, queryParams map[string]string, req interface{}, response interface{}) error {
	e := ErrorResponse{}
	resp, err := c.c.R().
		SetQueryParams(queryParams).
		SetHeader("X-Request-Id", uuid.Must(uuid.NewV4()).String()).
		SetBody(req).
		SetResult(response).
		SetError(&(e.IdmError)).
		Execute(method, url)
	if err != nil {
		return err
	}
	if resp.IsError() {
		e.StatusCode = resp.StatusCode()
		return e
	}
	return nil
}

func (c *Client) RequestRole(system, path, user, comment string) (roleID uint64, err error) {
	req := &RequestRoleReq{
		System:  system,
		Path:    path,
		User:    user,
		Comment: comment,
	}
	rsp := &RequestRoleRsp{}
	if err = c.doRequest("rolerequests/", resty.MethodPost, nil, req, rsp); err != nil {
		return
	}
	return rsp.Role.ID, nil
}

func (c *Client) GetRole(roleID uint64) (*Role, error) {
	rsp := &Role{}
	if err := c.doRequest(fmt.Sprintf("roles/%v/", roleID), resty.MethodGet, nil, nil, rsp); err != nil {
		return nil, err
	}
	return rsp, nil
}
