package idm

import (
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"

	ilog "a.yandex-team.ru/infra/infractl/internal/log"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/ptr"
)

func makeTime(t string) *time.Time {
	val, _ := time.Parse(time.RFC3339, t)
	return &val
}

func Test_getRole(t *testing.T) {
	svcLog := ilog.ConfigureZapLogger(log.DebugLevel, true)

	tests := []struct {
		name         string
		statusCode   int
		response     string
		expectedRole *Role
		wantError    assert.ErrorAssertionFunc
	}{
		{
			name:         "not found",
			statusCode:   404,
			response:     `{"error_code": "NOT_FOUND", "message": "No Role matches the given query."}`,
			expectedRole: nil,
			wantError: func(t assert.TestingT, err error, i ...interface{}) bool {
				return assert.ErrorIs(t, err, ErrorResponse{
					IdmError: IdmError{
						ErrorCode: "NOT_FOUND",
						Message:   "No Role matches the given query.",
					},
					StatusCode: 404,
				})
			},
		},
		{
			name:       "valid role",
			statusCode: 200,
			response: `{
				"added": "2022-07-11T02:48:55.240566+00:00",
				"expire_at": null,
				"fields_data": null,
				"granted_at": "2022-07-11T02:48:59.551839+00:00",
				"group": null,
				"human": "Роль: поиск по вики",
				"human_short": "поиск по вики",
				"human_state": "Выдана",
				"id": 77777777,
				"is_active": true,
				"is_public": null,
				"parent": null,
				"permissions": {
					"can_be_approved": false,
					"can_be_deprived": true,
					"can_be_poked_if_failed": false,
					"can_be_rerequested": false
				},
				"ref_count": 0,
				"review_at": null,
				"review_date": null,
				"review_days": null,
				"role_request": null,
				"state": "granted",
				"system_specific": null,
				"ttl_date": null,
				"ttl_days": null,
				"updated": "2022-07-11T02:48:59.551839+00:00",
				"user": {
					"date_joined": "2022-07-11",
					"email": "andrushka@yandex-team.ru",
					"fired_at": null,
					"full_name": "Андрей Чесноков",
					"is_active": true,
					"is_robot": false,
					"notify_responsibles": false,
					"position": "Менеджер холодных продаж (1 линия)",
					"sex": "M",
					"type": "user",
					"username": "andrushka"
				},
				"with_external": true,
				"with_inheritance": true,
				"with_robots": true,
				"without_hold": false
			}`,
			expectedRole: &Role{
				Added:       makeTime("2022-07-11T02:48:55.240566+00:00"),
				GrantedAt:   makeTime("2022-07-11T02:48:59.551839+00:00"),
				Human:       "Роль: поиск по вики",
				HumanShort:  "поиск по вики",
				HumanState:  "Выдана",
				ID:          77777777,
				IsActive:    true,
				Permissions: &Permissions{CanBeDeprived: true},
				State:       "granted",
				Updated:     makeTime("2022-07-11T02:48:59.551839+00:00"),
				User: &User{
					DateJoined: "2022-07-11",
					Email:      "andrushka@yandex-team.ru",
					FullName:   "Андрей Чесноков",
					IsActive:   true,
					Position:   "Менеджер холодных продаж (1 линия)",
					Sex:        ptr.String("M"),
					Type:       "user",
					Username:   "andrushka",
				},
				WithExternal:    true,
				WithInheritance: true,
				WithRobots:      true,
			},
			wantError: func(t assert.TestingT, err error, i ...interface{}) bool {
				return assert.NoError(t, err)
			},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.Header().Set("Content-Type", "application/json")
				w.WriteHeader(tt.statusCode)
				_, _ = w.Write([]byte(tt.response))
			}))
			defer ts.Close()

			c := NewClient(WithLogger(svcLog), WithURL(ts.URL))

			role, err := c.GetRole(12345)
			if role != nil {
				role.Data = nil
				role.FieldsData = nil
				role.Node = nil
				role.Parent = nil
				role.RoleRequest = nil
				role.System = nil
				role.SystemSpecific = nil
			}
			assert.Equal(t, tt.expectedRole, role)
			tt.wantError(t, err)
		})
	}
}
