package controllers

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/go-logr/logr"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/timestamppb"
	"k8s.io/apimachinery/pkg/api/errors"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/predicate"

	"a.yandex-team.ru/infra/awacs/clients/go/awacs"
	awacsapi "a.yandex-team.ru/infra/awacs/proto"
	"a.yandex-team.ru/infra/infractl/controllers/awacs/api/backend/proto_v1"
	v1 "a.yandex-team.ru/infra/infractl/controllers/awacs/api/backend/v1"
	"a.yandex-team.ru/infra/infractl/util/awacsutil"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

// AwacsBackendReconciler reconciles an AwacsBackend object
type AwacsBackendReconciler struct {
	client.Client
	TvmClient *tvmauth.Client
	YavClient *httpyav.Client
	TvmID     uint64
}

func (r *AwacsBackendReconciler) GetTvmClient() *tvmauth.Client {
	return r.TvmClient
}

func (r *AwacsBackendReconciler) GetYavClient() *httpyav.Client {
	return r.YavClient
}

func (r *AwacsBackendReconciler) GetK8sClient() client.Client {
	return r
}

func (r *AwacsBackendReconciler) statusEqual(old, new *proto_v1.Status) bool {
	return proto.Equal(old.BackendStatus, new.BackendStatus) &&
		proto.Equal(old.BackendL3Status, new.BackendL3Status) &&
		proto.Equal(old.BackendDnsStatus, new.BackendDnsStatus) &&
		proto.Equal(old.BackendResolverStatus, new.BackendResolverStatus)
}

func (r *AwacsBackendReconciler) ProcessBackend(
	ctx context.Context,
	req ctrl.Request,
	log logr.Logger,
	backend *v1.AwacsBackend,
) (*proto_v1.Status, *awacsapi.BackendSpec, string, error) {
	nameParts := strings.Split(req.Name, "--")
	if len(nameParts) != 2 {
		return nil, nil, "", fmt.Errorf("backend name does not match <namespace>--<id> pattern")
	}
	awacsNamespace, awacsBackend := nameParts[0], nameParts[1]

	token, err := getAwacsToken(r, ctx, log, req.Namespace, r.TvmID)
	if err != nil {
		return nil, nil, "", err
	}

	awacsClient := awacs.NewClient(awacs.WithToken(token))
	backendInfo, err := awacsutil.GetBackend(ctx, awacsClient, awacsNamespace, awacsBackend)
	var savedSpec *awacsapi.BackendSpec
	savedVersion := backend.Status.GetSyncStatus().GetLastAppliedAwacsVersion()

	if err != nil {
		return nil, nil, "", fmt.Errorf("failed to fetch awacs backend: %w", err)
	}
	if backendInfo == nil {
		log.Info("Backend needs creation")

		createRequest := &awacsapi.CreateBackendRequest{
			Meta: &awacsapi.BackendMeta{
				Id:          awacsBackend,
				NamespaceId: awacsNamespace,
				Comment:     fmt.Sprintf("Created by infractl from %v generation %v", req.NamespacedName, backend.Generation),
				Auth:        &awacsapi.Auth{Type: awacsapi.Auth_STAFF},
			},
			Spec: backend.Spec.BackendSpec,
		}
		createResponse, err := awacsClient.CreateBackend(ctx, createRequest)
		if err != nil {
			return nil, nil, "", fmt.Errorf("failed to create awacs backend: %w", err)
		}
		savedSpec = createResponse.Backend.Spec
		savedVersion = createResponse.Backend.Meta.Version
		backendInfo, err = awacsutil.GetBackend(ctx, awacsClient, awacsNamespace, awacsBackend)
		if err != nil {
			return nil, savedSpec, "", fmt.Errorf("failed to fetch created backend: %w", err)
		}
	} else {
		if backend.Status.SyncStatus.ObservedGeneration == backend.Generation &&
			proto.Equal(backendInfo.Spec, backend.Status.GetObservedTargetSpec()) {
			log.Info("spec is not changed")
		} else {
			updateRequest := &awacsapi.UpdateBackendRequest{
				Spec: backend.Spec.BackendSpec,
				Meta: &awacsapi.BackendMeta{
					Id:          awacsBackend,
					NamespaceId: awacsNamespace,
					Version:     backendInfo.Meta.Version,
					Comment:     fmt.Sprintf("Updated by infractl from %v generation %v", req.NamespacedName, backend.Generation),
				},
			}
			updateResponse, err := awacsClient.UpdateBackend(ctx, updateRequest)
			if err != nil {
				return nil, nil, "", fmt.Errorf("failed to update awacs backend: %w", err)
			}
			log.V(2).Info("backend updated", "response", updateResponse)
			savedSpec = updateResponse.Backend.Spec
			savedVersion = updateResponse.Backend.Meta.Version
			backendInfo, err = awacsutil.GetBackend(ctx, awacsClient, awacsNamespace, awacsBackend)
			if err != nil {
				return nil, savedSpec, "", fmt.Errorf("failed to fetch updated backend: %w", err)
			}
		}
	}

	backendStatus := &proto_v1.Status{}
	for _, status := range backendInfo.Statuses {
		if status.Id == savedVersion {
			backendStatus.BackendStatus = status
			break
		}
	}
	for _, status := range backendInfo.L3Statuses {
		if status.Id == savedVersion {
			backendStatus.BackendL3Status = status
			break
		}
	}
	for _, status := range backendInfo.DnsRecordStatuses {
		if status.Id == savedVersion {
			backendStatus.BackendDnsStatus = status
			break
		}
	}
	backendStatus.BackendResolverStatus = backendInfo.ResolverStatus

	return backendStatus, savedSpec, savedVersion, nil
}

func (r *AwacsBackendReconciler) Reconcile(ctx context.Context, req ctrl.Request) (ctrl.Result, error) {
	log := log.FromContext(ctx)
	log.Info("reconcile starting")
	defer log.Info("reconcile finished")

	result := ctrl.Result{
		RequeueAfter: time.Second * 60,
	}

	backend := &v1.AwacsBackend{}
	if err := r.Get(ctx, req.NamespacedName, backend); err != nil {
		log.Error(err, "unable to fetch backend")
		err = client.IgnoreNotFound(err)
		if errors.IsNotFound(err) {
			result = ctrl.Result{}
		}
		return result, nil
	}

	status, spec, version, err := r.ProcessBackend(ctx, req, log, backend)
	statusChanged := false
	if status != nil && !r.statusEqual(status, backend.Status) {
		statusChanged = true
		backend.Status.BackendStatus = status.BackendStatus
		backend.Status.BackendL3Status = status.BackendL3Status
		backend.Status.BackendDnsStatus = status.BackendDnsStatus
		backend.Status.BackendResolverStatus = status.BackendResolverStatus
		log.Info("status changed")
	}
	if spec != nil && !proto.Equal(spec, backend.Status.ObservedTargetSpec) {
		statusChanged = true
		backend.Status.ObservedTargetSpec = spec
		log.Info("spec changed")
	}
	if backend.Status.SyncStatus == nil {
		backend.Status.SyncStatus = &proto_v1.SyncStatus{}
	}
	if err != nil {
		log.Error(err, "failed to process backend")

		e := &ypapi.TCondition{
			Status:             ypapi.EConditionStatus_CS_TRUE,
			Reason:             fmt.Sprintf("%t", err),
			Message:            err.Error(),
			LastTransitionTime: timestamppb.Now(),
		}
		if !proto.Equal(backend.Status.SyncStatus.Error, e) {
			statusChanged = true
		}
		backend.Status.SyncStatus.Error = e
		backend.Status.SyncStatus.Success = nil
	} else if statusChanged || backend.Status.SyncStatus.Success == nil {
		backend.Status.SyncStatus.Error = nil
		s := &ypapi.TCondition{
			Status:             ypapi.EConditionStatus_CS_TRUE,
			Reason:             "",
			Message:            "",
			LastTransitionTime: timestamppb.Now(),
		}
		if !proto.Equal(backend.Status.SyncStatus.Success, s) {
			statusChanged = true
		}
		backend.Status.SyncStatus.Success = s
		backend.Status.SyncStatus.LastAppliedAwacsVersion = version
	}
	backend.Status.SyncStatus.ObservedGeneration = backend.Generation

	if statusChanged {
		log.Info("updating status")
		if err = r.Status().Update(ctx, backend); err != nil {
			log.Error(err, "failed to write status")
		}
	}

	return result, nil
}

// SetupWithManager sets up the controller with the Manager.
func (r *AwacsBackendReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&v1.AwacsBackend{}).
		WithEventFilter(predicate.GenerationChangedPredicate{}).
		Complete(r)
}
