package controllers

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/go-logr/logr"
	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/timestamppb"
	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/types"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/predicate"

	"a.yandex-team.ru/infra/awacs/clients/go/awacs"
	awacsapi "a.yandex-team.ru/infra/awacs/proto"
	"a.yandex-team.ru/infra/infractl/controllers/awacs/api/upstream/proto_v1"
	v1 "a.yandex-team.ru/infra/infractl/controllers/awacs/api/upstream/v1"
	"a.yandex-team.ru/infra/infractl/util/awacsutil"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

// AwacsUpstreamReconciler reconciles an AwacsUpstream object
type AwacsUpstreamReconciler struct {
	client.Client
	TvmClient *tvmauth.Client
	YavClient *httpyav.Client
	TvmID     uint64
}

func (r *AwacsUpstreamReconciler) GetTvmClient() *tvmauth.Client {
	return r.TvmClient
}

func (r *AwacsUpstreamReconciler) GetYavClient() *httpyav.Client {
	return r.YavClient
}

func (r *AwacsUpstreamReconciler) GetK8sClient() client.Client {
	return r
}

func (r *AwacsUpstreamReconciler) createIfNotExists(ctx context.Context, log logr.Logger, ns *corev1.Namespace, toFind client.Object, toCreate client.Object) error {
	err := r.Get(ctx, types.NamespacedName{Namespace: toCreate.GetNamespace(), Name: toCreate.GetName()}, toFind)
	if err == nil {
		return nil
	}
	if !errors.IsNotFound(err) {
		return fmt.Errorf("failed to get: %w", err)
	}
	if err = controllerutil.SetControllerReference(ns, toCreate, r.Scheme()); err != nil {
		return fmt.Errorf("failed to set controller reference: %w", err)
	}
	err = r.Create(ctx, toCreate)
	if err != nil && !errors.IsAlreadyExists(err) {
		return fmt.Errorf("failed to create: %w", err)
	}
	return nil
}

func (r *AwacsUpstreamReconciler) ProcessUpstream(
	ctx context.Context,
	req ctrl.Request,
	log logr.Logger,
	upstream *v1.AwacsUpstream,
) (*awacsapi.UpstreamRevisionStatusPerBalancer, *awacsapi.UpstreamSpec, string, error) {
	nameParts := strings.Split(req.Name, "--")
	if len(nameParts) != 2 {
		return nil, nil, "", fmt.Errorf("upstream name does not match <namespace>--<id> pattern")
	}

	awacsNamespace, awacsUpstream := nameParts[0], nameParts[1]

	token, err := getAwacsToken(r, ctx, log, req.Namespace, r.TvmID)
	if err != nil {
		return nil, nil, "", err
	}

	awacsClient := awacs.NewClient(awacs.WithToken(token))
	upstreamInfo, err := awacsutil.GetUpstream(ctx, awacsClient, awacsNamespace, awacsUpstream)
	var savedSpec *awacsapi.UpstreamSpec
	savedVersion := upstream.Status.GetSyncStatus().GetLastAppliedAwacsVersion()

	if err != nil {
		return nil, nil, "", fmt.Errorf("failed to fetch awacs upstream: %w", err)
	}
	if upstreamInfo == nil {
		log.Info("Upstream needs creation")

		createRequest := &awacsapi.CreateUpstreamRequest{
			Meta: &awacsapi.UpstreamMeta{
				Id:          awacsUpstream,
				NamespaceId: awacsNamespace,
				Comment:     fmt.Sprintf("Created by infractl from %v generation %v", req.NamespacedName, upstream.Generation),
				Annotations: map[string]string{
					"managed-by": "infractl",
				},
				Auth: &awacsapi.Auth{Type: awacsapi.Auth_STAFF},
			},
			Spec: upstream.Spec.UpstreamSpec,
		}
		createResponse, err := awacsClient.CreateUpstream(ctx, createRequest)
		if err != nil {
			return nil, nil, "", fmt.Errorf("failed to create awacs upstream: %w", err)
		}
		savedSpec = createResponse.Upstream.Spec
		savedVersion = createResponse.Upstream.Meta.Version
		upstreamInfo, err = awacsutil.GetUpstream(ctx, awacsClient, awacsNamespace, awacsUpstream)
		if err != nil {
			return nil, savedSpec, "", fmt.Errorf("failed to fetch created upstream: %w", err)
		}
	} else {
		if upstream.Status.SyncStatus.ObservedGeneration == upstream.Generation &&
			proto.Equal(upstreamInfo.Spec, upstream.Status.GetObservedTargetSpec()) {
			log.Info("spec is not changed")
		} else {
			updateRequest := &awacsapi.UpdateUpstreamRequest{
				Spec: upstream.Spec.UpstreamSpec,
				Meta: &awacsapi.UpstreamMeta{
					Id:          awacsUpstream,
					NamespaceId: awacsNamespace,
					Version:     upstreamInfo.Meta.Version,
					Comment:     fmt.Sprintf("Updated by infractl from %v generation %v", req.NamespacedName, upstream.Generation),
				},
			}
			updateResponse, err := awacsClient.UpdateUpstream(ctx, updateRequest)
			if err != nil {
				return nil, nil, "", fmt.Errorf("failed to update awacs upstream: %w", err)
			}
			log.V(2).Info("upstream updated", "response", updateResponse)
			savedSpec = updateResponse.Upstream.Spec
			savedVersion = updateResponse.Upstream.Meta.Version
			upstreamInfo, err = awacsutil.GetUpstream(ctx, awacsClient, awacsNamespace, awacsUpstream)
			if err != nil {
				return nil, savedSpec, "", fmt.Errorf("failed to fetch updated upstream: %w", err)
			}
		}
	}

	var upstreamStatus *awacsapi.UpstreamRevisionStatusPerBalancer
	for _, status := range upstreamInfo.Statuses {
		if status.Id == savedVersion {
			upstreamStatus = status
			break
		}
	}

	return upstreamStatus, savedSpec, savedVersion, nil
}

func (r *AwacsUpstreamReconciler) Reconcile(ctx context.Context, req ctrl.Request) (ctrl.Result, error) {
	log := log.FromContext(ctx)
	log.Info("reconcile starting")
	defer log.Info("reconcile finished")

	result := ctrl.Result{
		RequeueAfter: time.Second * 60,
	}

	upstream := &v1.AwacsUpstream{}
	if err := r.Get(ctx, req.NamespacedName, upstream); err != nil {
		log.Error(err, "unable to fetch upstream")
		err = client.IgnoreNotFound(err)
		if errors.IsNotFound(err) {
			result = ctrl.Result{}
		}
		return result, nil
	}

	status, spec, version, err := r.ProcessUpstream(ctx, req, log, upstream)
	statusChanged := false
	if status != nil && !proto.Equal(status, upstream.Status.UpstreamStatus) {
		statusChanged = true
		upstream.Status.UpstreamStatus = status
		log.Info("status changed")
	}
	if spec != nil && !proto.Equal(spec, upstream.Status.ObservedTargetSpec) {
		statusChanged = true
		upstream.Status.ObservedTargetSpec = spec
		log.Info("spec changed")
	}
	if upstream.Status.SyncStatus == nil {
		upstream.Status.SyncStatus = &proto_v1.SyncStatus{}
	}
	if err != nil {
		log.Error(err, "failed to process upstream")

		e := &ypapi.TCondition{
			Status:             ypapi.EConditionStatus_CS_TRUE,
			Reason:             fmt.Sprintf("%t", err),
			Message:            err.Error(),
			LastTransitionTime: timestamppb.Now(),
		}
		if !proto.Equal(upstream.Status.SyncStatus.Error, e) {
			statusChanged = true
		}
		upstream.Status.SyncStatus.Error = e
		upstream.Status.SyncStatus.Success = nil
	} else if statusChanged || upstream.Status.SyncStatus.Success == nil {
		upstream.Status.SyncStatus.Error = nil
		s := &ypapi.TCondition{
			Status:             ypapi.EConditionStatus_CS_TRUE,
			Reason:             "",
			Message:            "",
			LastTransitionTime: timestamppb.Now(),
		}
		if !proto.Equal(upstream.Status.SyncStatus.Success, s) {
			statusChanged = true
		}
		upstream.Status.SyncStatus.Success = s
		upstream.Status.SyncStatus.LastAppliedAwacsVersion = version
	}
	upstream.Status.SyncStatus.ObservedGeneration = upstream.Generation

	if statusChanged {
		log.Info("updating status")
		if err = r.Status().Update(ctx, upstream); err != nil {
			log.Error(err, "failed to write status")
		}
	}

	return result, nil
}

// SetupWithManager sets up the controller with the Manager.
func (r *AwacsUpstreamReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&v1.AwacsUpstream{}).
		WithEventFilter(predicate.GenerationChangedPredicate{}).
		Complete(r)
}
