package main

import (
	"os"

	"github.com/spf13/pflag"
	"k8s.io/client-go/kubernetes/scheme"
	// Import all Kubernetes client auth plugins (e.g. Azure, GCP, OIDC, etc.)
	// to ensure that exec-entrypoint and run can make use of them.
	_ "k8s.io/client-go/plugin/pkg/client/auth"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/healthz"

	"a.yandex-team.ru/infra/infractl/controllers/awacs/api"
	"a.yandex-team.ru/infra/infractl/controllers/awacs/controllers"
	"a.yandex-team.ru/infra/infractl/internal/log"
	_ "a.yandex-team.ru/infra/infractl/internal/scheme"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

var (
	setupLog = ctrl.Log.WithName("setup")
)

func main() {
	configFile := pflag.String("config", "",
		"The controller will load its initial configuration from this file. "+
			"Omit this flag to use the default configuration values. "+
			"Command-line flags override configuration from this file.")

	logopts := log.Bind(pflag.CommandLine)

	pflag.Parse()

	ctrl.SetLogger(logopts.ConfigureLogger())

	ctrlConfig := api.ControllerConfig{}
	options := ctrl.Options{Scheme: scheme.Scheme}
	var err error
	if *configFile != "" {
		options, err = options.AndFrom(ctrl.ConfigFile().AtPath(*configFile).OfKind(&ctrlConfig))
		if err != nil {
			setupLog.Error(err, "unable to load config file")
			os.Exit(1)
		}
	}

	mgr, err := ctrl.NewManager(ctrl.GetConfigOrDie(), options)

	if err != nil {
		setupLog.Error(err, "unable to start manager", "config", ctrl.GetConfigOrDie())
		os.Exit(1)
	}

	tvmSecret := os.Getenv("TVM_SECRET")
	if tvmSecret == "" {
		setupLog.Error(err, "unable to get TVM_SECRET env variable")
		os.Exit(1)
	}
	tvmSettings := tvmauth.TvmAPISettings{
		SelfID: tvm.ClientID(ctrlConfig.Options.TvmID),
		ServiceTicketOptions: tvmauth.NewAliasesOptions(
			tvmSecret,
			map[string]tvm.ClientID{
				"yav": tvm.ClientID(ctrlConfig.Options.YavTvmID),
			},
		),
	}
	tvmc, err := tvmauth.NewAPIClient(tvmSettings, &nop.Logger{})
	if err != nil {
		setupLog.Error(err, "unable to create tvm client")
		os.Exit(1)
	}

	yavc, err := httpyav.NewClient()
	if err != nil {
		setupLog.Error(err, "unable to create yav client")
		os.Exit(1)
	}

	upstreamReconciler := &controllers.AwacsUpstreamReconciler{
		Client:    mgr.GetClient(),
		TvmClient: tvmc,
		YavClient: yavc,
		TvmID:     ctrlConfig.Options.TvmID,
	}

	backendReconciler := &controllers.AwacsBackendReconciler{
		Client:    mgr.GetClient(),
		TvmClient: tvmc,
		YavClient: yavc,
		TvmID:     ctrlConfig.Options.TvmID,
	}

	/* no unistats yet
	if err = mgr.AddMetricsExtraHandler(ctrlConfig.Unistat.Path, reconciler.Stats); err != nil {
		setupLog.Error(err, "unable to setup unistat", "controller", "Awacs")
		os.Exit(1)
	}
	*/

	if err = upstreamReconciler.SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "AwacsUpstream")
		os.Exit(1)
	}
	if err = backendReconciler.SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "AwacsBackend")
		os.Exit(1)
	}

	if err = mgr.AddHealthzCheck("healthz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up health check")
		os.Exit(1)
	}
	if err = mgr.AddReadyzCheck("readyz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up ready check")
		os.Exit(1)
	}

	setupLog.Info("starting manager")
	if err = mgr.Start(ctrl.SetupSignalHandler()); err != nil {
		setupLog.Error(err, "problem running manager")
		os.Exit(1)
	}
}
