package v1

import (
	"fmt"

	"google.golang.org/protobuf/proto"

	v1proto "a.yandex-team.ru/infra/infractl/controllers/deploy/api/common/proto_v1"
	"a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/proto_v1"
	"a.yandex-team.ru/infra/infractl/controllers/deploy/ypproject"
	ctrls "a.yandex-team.ru/infra/infractl/controllers/internal"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
)

const (
	ProjectFqidAnnotation = "project.infractl.k.yandex-team.ru/fqid"
)

func (o *DeployProject) GetObjectType() ypapi.EObjectType {
	return yp.ObjectTypeProject
}

func (o *DeployProject) GetReadableObjectType() string {
	return "Project"
}

func (o *DeployProject) GetAccountID() string {
	return o.Spec.GetProjectSpec().GetAccountId()
}

func (o *DeployProject) GetSyncStatus() *v1proto.SyncStatus {
	if o.Status == nil {
		o.Status = &proto_v1.Status{}
	}
	if o.Status.GetSyncStatus() == nil {
		o.Status.SyncStatus = &v1proto.SyncStatus{}
	}
	return o.Status.GetSyncStatus()
}

func (o *DeployProject) IsStatusEqualTo(kObj interfaces.KubernetesObject) (bool, error) {
	var equal bool
	kProject, ok := kObj.(*DeployProject)
	if !ok {
		return equal, fmt.Errorf("failed to cast object to k8s project")
	}
	return proto.Equal(kProject.Status, o.Status), nil
}

func (o *DeployProject) SpecNewer(generation int64) bool {
	return ctrls.SpecNewer(o, generation)
}

func (o *DeployProject) MakeURL(template string) string {
	return fmt.Sprintf(template, "projects", o.GetName())
}

func (o *DeployProject) FillDeployObject(response *yp.GetObjectResponse) (interfaces.DeployObject, error) {
	project := &ypapi.TProject{}
	if err := response.Fill(project); err != nil {
		return nil, err
	}
	dObj := ypproject.NewYPProject(project, response.Timestamps()[1])
	return dObj, nil
}

func (o *DeployProject) SetStatus(dObj interfaces.DeployObject) error {
	ypPrj, ok := dObj.GetYPObject().(*ypapi.TProject)
	if !ok {
		return fmt.Errorf("failed to cast deploy project to YP project")
	}
	if o.Status == nil {
		o.Status = &proto_v1.Status{}
	}
	o.Status.ProjectStatus = ypPrj.GetStatus().DeepCopy()
	return nil
}

func (o *DeployProject) ValidateDeployObject(_ interfaces.DeployObject) (string, error) {
	return "", nil
}

func (o *DeployProject) SetFqid(fqid string) {
	if o.Annotations == nil {
		o.Annotations = make(map[string]string)
	}
	o.Annotations[ProjectFqidAnnotation] = fqid
}
