package v1

import (
	"fmt"
	"strings"

	"google.golang.org/protobuf/proto"
	"k8s.io/apimachinery/pkg/util/sets"

	commonv1proto "a.yandex-team.ru/infra/infractl/controllers/deploy/api/common/proto_v1"
	protov1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/proto_v1"
	"a.yandex-team.ru/infra/infractl/controllers/deploy/ypstage"
	ctrls "a.yandex-team.ru/infra/infractl/controllers/internal"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
)

const (
	RevisionCommentAnnotation = "deploystage.infractl.k.yandex-team.ru/revisionComment"
)

func (o *DeployStage) setYPStatus(dObj interfaces.DeployObject) error {
	ypStage, ok := dObj.GetYPObject().(*ypapi.TStage)
	if !ok {
		return fmt.Errorf("failed to cast deploy stage to YP stage")
	}
	o.Status.StageStatus = ypStage.GetStatus().DeepCopy()
	return nil
}

func (o *DeployStage) setAppliedStatus(dObj interfaces.DeployObject) error {
	ypStage, ok := dObj.GetYPObject().(*ypapi.TStage)
	if !ok {
		return fmt.Errorf("failed to cast deploy stage to YP stage")
	}
	if o.Status.GetAppliedDeployRevisions() == nil {
		o.Status.AppliedDeployRevisions = &protov1.AppliedDeployRevisions{}
	}
	o.Status.AppliedDeployRevisions.DeployUnits = make(map[string]uint32, len(ypStage.GetSpec().GetDeployUnits()))
	o.Status.AppliedDeployRevisions.DynamicResources = make(map[string]uint64, len(ypStage.GetSpec().GetDynamicResources()))
	for duName, du := range ypStage.GetSpec().GetDeployUnits() {
		o.Status.AppliedDeployRevisions.DeployUnits[duName] = du.GetRevision()
	}
	for drName, dr := range ypStage.GetSpec().GetDynamicResources() {
		o.Status.AppliedDeployRevisions.DynamicResources[drName] = dr.GetDynamicResource().GetRevision()
	}
	return nil
}

func (o *DeployStage) GetObjectType() ypapi.EObjectType {
	return yp.ObjectTypeStage
}

func (o *DeployStage) GetReadableObjectType() string {
	return "Stage"
}

func (o *DeployStage) GetAccountID() string {
	return o.Spec.GetStageSpec().GetAccountId()
}

func (o *DeployStage) GetSyncStatus() *commonv1proto.SyncStatus {
	if o.Status.GetSyncStatus() == nil {
		o.Status.SyncStatus = &commonv1proto.SyncStatus{}
	}
	return o.Status.GetSyncStatus()
}

func (o *DeployStage) IsStatusEqualTo(kObj interfaces.KubernetesObject) (bool, error) {
	var equal bool
	kStage, ok := kObj.(*DeployStage)
	if !ok {
		return equal, fmt.Errorf("failed to cast object to k8s stage")
	}
	return proto.Equal(kStage.Status, o.Status), nil
}

func (o *DeployStage) SpecNewer(generation int64) bool {
	return ctrls.SpecNewer(o, generation)
}

func (o *DeployStage) MakeURL(template string) string {
	return fmt.Sprintf(template, "stages", o.GetName())
}

func (o *DeployStage) FillDeployObject(response *yp.GetObjectResponse) (interfaces.DeployObject, error) {
	stage := &ypapi.TStage{}
	if err := response.Fill(stage); err != nil {
		return nil, err
	}
	// We use timestamp to enable CAS in updateObject
	dObj := ypstage.NewYPStage(stage, response.Timestamps()[1], o.GetAnnotations()[RevisionCommentAnnotation])
	return dObj, nil
}

func (o *DeployStage) ValidateDeployObject(dObj interfaces.DeployObject) (string, error) {
	ypStage, ok := dObj.GetYPObject().(*ypapi.TStage)
	if !ok {
		msg := "failed to cast deploy stage to YP stage"
		return strings.ToTitle(msg), fmt.Errorf(msg)
	}
	projectID := ypStage.GetMeta().GetProjectId()
	if projectID != o.GetNamespace() {
		msg := fmt.Sprintf("namespace name %q does not match project_id %v in YP", o.Namespace, projectID)
		return strings.ToTitle(msg), fmt.Errorf(msg)
	}
	return "", nil
}

func (o *DeployStage) SetStatus(dObj interfaces.DeployObject) error {
	if err := o.setYPStatus(dObj); err != nil {
		return err
	}
	if err := o.setAppliedStatus(dObj); err != nil {
		return err
	}
	return nil
}

func (o *DeployStage) SetFqid(fqid string) {
	if o.Annotations == nil {
		o.Annotations = make(map[string]string)
	}
	o.Annotations[FqidAnnotation] = fqid
}

func (o *DeployStage) CollectSecrets() sets.String {
	result := sets.String{}
	for _, deployUnit := range o.GetSpec().GetStageSpec().GetDeployUnits() {
		podSpec := dclient.GеtPodSpec(deployUnit)
		// Deprecated Secrets ain't processed, just forget them
		for _, secretRef := range podSpec.GetSecretRefs() {
			result.Insert(secretRef.GetSecretId())
		}
	}
	return result
}
