package controllers

import (
	"context"

	corev1 "k8s.io/api/core/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"

	commonv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/common/v1"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/yp/go/yp"
)

// DeployReconciler is base object for DeployStageReconciler and DeployProjectReconciler
type DeployReconciler struct {
	client.Client
	Stats           *Stats
	DefaultYPClient *yp.Client
	TvmClient       *tvmauth.Client
	YavClient       *httpyav.Client
	Options         commonv1.ReconcilerOptions
}

func (r *DeployReconciler) GetK8sClient() client.Client {
	return r
}
func (r *DeployReconciler) GetTvmClient() *tvmauth.Client {
	return r.TvmClient
}

func (r *DeployReconciler) GetYavClient() *httpyav.Client {
	return r.YavClient
}

func (r *DeployReconciler) ChooseYpClient(ctx context.Context, namespace corev1.Namespace) (*yp.Client, error) {
	if r.Options.DevMode {
		return r.DefaultYPClient, nil
	} else {
		return dclient.InitYpClientWithToken(ctx, r, &namespace, r.Options.YpClusterName, r.Options.TvmID)
	}
}

// DeleteDeployObject goes to Yandex Deploy and deletes Deploy Object that is related to kObj
func (r *DeployReconciler) DeleteDeployObject(
	ctx context.Context,
	kObj interfaces.KubernetesObject,
	deployClient *dclient.DeployClient,
	ypClient *yp.Client,
	meta interface{},
) error {
	req := yp.RemoveObjectRequest{
		ObjectType:        kObj.GetObjectType(),
		Meta:              meta,
		IgnoreNonexistent: false,
	}
	_, err := ypClient.RemoveObject(ctx, req)
	return err
}
