package controllers

import (
	"context"
	"fmt"
	"time"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/types"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	projectv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/v1"
	"a.yandex-team.ru/infra/infractl/controllers/deploy/ypproject"
	ctrls "a.yandex-team.ru/infra/infractl/controllers/internal"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type deployProjectMaker struct {
	accountID string
}

func (m *deployProjectMaker) Make(kObj interfaces.KubernetesObject, prevDobj interfaces.DeployObject) (interfaces.DeployObject, error) {
	dProject := kObj.(*projectv1.DeployProject)
	spec := dProject.Spec.GetProjectSpec().DeepCopy()
	if spec == nil {
		spec = &ypapi.TProjectSpec{}
	}
	spec.AccountId = m.accountID
	return ypproject.NewYPProject(
		&ypapi.TProject{
			Meta: &ypapi.TProjectMeta{
				Id: dProject.GetName(),
			},
			Spec: spec,
		},
		0,
	), nil
}

// DeployProjectReconciler reconciles a DeployProject object
type DeployProjectReconciler struct {
	DeployReconciler
	ABCClient *abc.Client
}

func (r *DeployProjectReconciler) getAccountID(namespace *corev1.Namespace) (string, error) {
	abcSlug := namespace.Labels[labels.ABC]
	if abcSlug == "" {
		return "", fmt.Errorf("failed to find %s label key in namespace %s", labels.ABC, namespace.Name)
	}
	abcID, err := r.ABCClient.GetServiceIDBySlug(abcSlug)
	if err != nil {
		return "", fmt.Errorf("failed get ABC-service ID by slug \"%s\": %w", abcSlug, err)
	}
	return fmt.Sprintf("abc:service:%d", abcID), nil
}

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.8.3/pkg/reconcile
func (r *DeployProjectReconciler) Reconcile(
	ctx context.Context,
	req ctrl.Request,
) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)
	log.Info("Syncing")
	defer log.Info("Sync finished")

	result = ctrl.Result{
		RequeueAfter: time.Second * 60,
	}

	kProject := &projectv1.DeployProject{}
	if err = r.Get(ctx, req.NamespacedName, kProject); err != nil {
		log.Error(err, "unable to fetch DeployProject")
		// TODO(torkve) should we remove project from YP here? and how?
		err = client.IgnoreNotFound(err)
		if err == nil {
			result = ctrl.Result{}
		}
		return
	}
	log = log.WithValues("generation", kProject.GetGeneration())

	reconciler := ctrls.Reconciler{
		TemplateURL:    r.Options.TemplateURL,
		StatusWriter:   r.Status(),
		UnistatUpdater: r.Stats,
		Client:         r.Client,
	}
	if err = func() error {
		var namespace corev1.Namespace
		if err = r.Get(ctx, types.NamespacedName{Namespace: "", Name: kProject.Namespace}, &namespace); err != nil {
			return fmt.Errorf("failed to fetch namespace %s: %w", kProject.Namespace, err)
		}

		userYPClient, err := r.ChooseYpClient(ctx, namespace)
		if err != nil {
			return fmt.Errorf("failed to instantiate YP client: %w", err)
		}

		var accID string
		accID, err = r.getAccountID(&namespace)
		if err != nil {
			return fmt.Errorf("failed to set ABC account id to project: %w", err)
		}

		deployClient := dclient.NewDeployClient(r.DefaultYPClient, userYPClient, r)
		if !kProject.ObjectMeta.DeletionTimestamp.IsZero() {
			if controllerutil.ContainsFinalizer(kProject, ctrls.BeforeDeleteFinalizer) {
				if err = r.DeleteDeployObject(ctx, kProject, deployClient, userYPClient, &ypapi.TProjectMeta{Fqid: kProject.GetFqid()}); err != nil {
					return err
				}
				controllerutil.RemoveFinalizer(kProject, ctrls.BeforeDeleteFinalizer)
				if err = r.Update(ctx, kProject); err != nil {
					return err
				}
			}
		}
		_ = reconciler.Reconcile(ctx, log, deployClient, kProject, &deployProjectMaker{accountID: accID})
		return nil
	}(); err != nil {
		if err = reconciler.UpdateStatus(ctx, log, kProject, kProject.DeepCopy(), ctrls.NewReasonedError(err, ctrls.ProcessObjectFailedReason)); err != nil {
			log.Error(err, "failed to update status")
		}
	}
	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *DeployProjectReconciler) SetupWithManager(mgr ctrl.Manager) error {
	nsIndexFunc := func(rawObj client.Object) []string {
		// Extract the Namespace name from the DeployProject, if one is provided
		kProject := rawObj.(*projectv1.DeployProject)
		if kProject.Namespace == "" {
			return nil
		}
		return []string{kProject.Namespace}
	}
	/*
		The `namespace` field must be indexed by the manager, so that we will be able to lookup `DeployProjects` by a referenced `Namespace` name.
		This will allow for quickly answer the question:
		- If Namespace _x_ is updated, which DeployProjects are affected?
	*/
	if err := mgr.GetFieldIndexer().IndexField(context.Background(), &projectv1.DeployProject{}, namespaceIndexField, nsIndexFunc); err != nil {
		return err
	}
	return ctrl.NewControllerManagedBy(mgr).
		For(&projectv1.DeployProject{}).
		Watches(
			&source.Kind{Type: &corev1.Namespace{}},
			handler.EnqueueRequestsFromMapFunc(func(namespace client.Object) []reconcile.Request { return []reconcile.Request{} }),
		).
		Complete(r)
}
