package controllers

import (
	"context"
	"fmt"
	"time"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/types"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"

	deployinfrav1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	"a.yandex-team.ru/infra/infractl/controllers/deploy/ypstage"
	ctrls "a.yandex-team.ru/infra/infractl/controllers/internal"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

const (
	namespaceIndexField = ".metadata.namespace"
	secretIndexField    = ".metadata.annotations.infractl-secret"
)

type deployStageMaker struct{}

func (m *deployStageMaker) Make(kObj interfaces.KubernetesObject, prevDobj interfaces.DeployObject) (interfaces.DeployObject, error) {
	dStage := kObj.(*deployinfrav1.DeployStage)
	return ypstage.NewYPStage(
		&ypapi.TStage{
			Meta: &ypapi.TStageMeta{
				Id:        kObj.GetName(),
				ProjectId: kObj.GetNamespace(),
			},
			Spec: dStage.Spec.GetStageSpec(),
		},
		0,
		kObj.GetAnnotations()[deployinfrav1.RevisionCommentAnnotation],
	), nil
}

// DeployStageReconciler reconciles a DeployStage object
type DeployStageReconciler struct {
	DeployReconciler
}

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.8.3/pkg/reconcile
func (r *DeployStageReconciler) Reconcile(
	ctx context.Context,
	req ctrl.Request,
) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)
	log.Info("Syncing")
	defer log.Info("Sync finished")

	result = ctrl.Result{
		RequeueAfter: time.Second * 60,
	}

	kStage := &deployinfrav1.DeployStage{}
	if err = r.Get(ctx, req.NamespacedName, kStage); err != nil {
		log.Error(err, "unable to fetch DeployStage")
		// TODO(torkve) should we remove stage from YP here? and how?
		err = client.IgnoreNotFound(err)
		if err == nil {
			result = ctrl.Result{}
		} else {
			r.Stats.SyncErrors.Update(1)
		}
		return
	}

	log = log.WithValues("generation", kStage.GetGeneration())

	reconciler := ctrls.Reconciler{
		TemplateURL:    r.Options.TemplateURL,
		StatusWriter:   r.Status(),
		UnistatUpdater: r.Stats,
		Client:         r.Client,
	}
	if err = func() error {
		var namespace corev1.Namespace
		if err = r.Get(ctx, types.NamespacedName{Namespace: "", Name: kStage.Namespace}, &namespace); err != nil {
			return fmt.Errorf("failed to fetch namespace %s: %w", kStage.Namespace, err)
		}

		userYPClient, err := r.ChooseYpClient(ctx, namespace)
		if err != nil {
			return fmt.Errorf("failed to instantiate YP client: %w", err)
		}

		deployClient := dclient.NewDeployClient(r.DefaultYPClient, userYPClient, r)

		if !kStage.ObjectMeta.DeletionTimestamp.IsZero() {
			if controllerutil.ContainsFinalizer(kStage, ctrls.BeforeDeleteFinalizer) {
				if err = r.DeleteDeployObject(ctx, kStage, deployClient, userYPClient, &ypapi.TStageMeta{Fqid: kStage.GetFqid()}); err != nil {
					return err
				}
				controllerutil.RemoveFinalizer(kStage, ctrls.BeforeDeleteFinalizer)
				if err = r.Update(ctx, kStage); err != nil {
					return err
				}
			}
		}
		_ = reconciler.Reconcile(ctx, log, deployClient, kStage, &deployStageMaker{})
		return nil
	}(); err != nil {
		if err = reconciler.UpdateStatus(ctx, log, kStage, kStage.DeepCopy(), ctrls.NewReasonedError(err, ctrls.ProcessObjectFailedReason)); err != nil {
			log.Error(err, "failed to update status")
		}
	}
	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *DeployStageReconciler) SetupWithManager(mgr ctrl.Manager) error {
	nsIndexFunc := func(rawObj client.Object) []string {
		// Extract the Namespace name from the DeployStage, if one is provided
		kStage := rawObj.(*deployinfrav1.DeployStage)
		if kStage.Namespace == "" {
			return nil
		}
		return []string{kStage.Namespace}
	}
	/*
		The `namespace` field must be indexed by the manager, so that we will be able to lookup `DeployStages` by a referenced `Namespace` name.
		This will allow for quickly answer the question:
		- If Namespace _x_ is updated, which DeployStages are affected?
	*/
	if err := mgr.GetFieldIndexer().IndexField(context.Background(), &deployinfrav1.DeployStage{}, namespaceIndexField, nsIndexFunc); err != nil {
		return err
	}
	/*
		The `secret` field must be indexed by the manager, so that we will be able to lookup `DeployStages` by a referenced `Secret` name.
		This will allow for quickly answer the question:
		- If Secret _x_ is updated, which DeployStages are affected?
	*/
	if err := mgr.GetFieldIndexer().IndexField(context.Background(), &deployinfrav1.DeployStage{}, secretIndexField, nsIndexFunc); err != nil {
		return err
	}
	return ctrl.NewControllerManagedBy(mgr).
		For(&deployinfrav1.DeployStage{}).
		Watches(
			&source.Kind{Type: &corev1.Namespace{}},
			handler.EnqueueRequestsFromMapFunc(func(namespace client.Object) []reconcile.Request { return []reconcile.Request{} }),
		).
		Watches(
			&source.Kind{Type: &corev1.Secret{}},
			handler.EnqueueRequestsFromMapFunc(func(namespace client.Object) []reconcile.Request { return []reconcile.Request{} }),
		).
		Complete(r)
}
