package controllers

import (
	"fmt"
	"net/http"

	"sigs.k8s.io/controller-runtime/pkg/log"

	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
)

// Stats implements common.UnistatUpdater
type Stats struct {
	SyncCount   *unistat.Numeric
	SyncErrors  *unistat.Numeric
	SyncLatency *unistat.Histogram
}

func (s *Stats) GetSyncLatency() *unistat.Histogram {
	return s.SyncLatency
}

func (s *Stats) UpdateSyncErrors(value float64) {
	s.SyncErrors.Update(value)
}

func (s *Stats) UpdateSyncCount(value float64) {
	s.SyncCount.Update(value)
}

func MakeStats(objectType string) *Stats {
	stats := &Stats{
		SyncCount:  unistat.NewNumeric(fmt.Sprintf("%s-sync-count", objectType), 1, aggr.Counter(), unistat.Sum),
		SyncErrors: unistat.NewNumeric(fmt.Sprintf("%s-sync-errors", objectType), 1, aggr.Counter(), unistat.Sum),
		SyncLatency: unistat.NewHistogram(fmt.Sprintf("%s-sync-latency", objectType), 1, aggr.Histogram(),
			[]float64{0, 25, 50, 75, 100, 500, 1000, 5000, 10000}),
	}
	unistat.Register(stats.SyncCount)
	unistat.Register(stats.SyncErrors)
	unistat.Register(stats.SyncLatency)

	return stats
}

func (s *Stats) ServeHTTP(rw http.ResponseWriter, req *http.Request) {
	bytes, err := unistat.MarshalJSON()
	if err != nil {
		http.Error(rw, err.Error(), http.StatusInternalServerError)
		return
	}

	rw.Header().Set("Content-Type", "application/json; charset=utf-8")
	if _, err := rw.Write(bytes); err != nil {
		log := log.FromContext(req.Context())
		log.Error(err, "Failed to send response")
	}
}
