package main

import (
	"os"

	"github.com/spf13/pflag"
	"k8s.io/client-go/kubernetes/scheme"
	// Import all Kubernetes client auth plugins (e.g. Azure, GCP, OIDC, etc.)
	// to ensure that exec-entrypoint and run can make use of them.
	_ "k8s.io/client-go/plugin/pkg/client/auth"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/healthz"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	commonv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/common/v1"
	"a.yandex-team.ru/infra/infractl/controllers/deploy/controllers"
	"a.yandex-team.ru/infra/infractl/internal/log"
	_ "a.yandex-team.ru/infra/infractl/internal/scheme"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
	"a.yandex-team.ru/yp/go/yp"
)

var (
	setupLog     = ctrl.Log.WithName("setup")
	envTokenName = "YP_TOKEN"
)

func main() {
	devMode := pflag.Bool("dev-mode", false,
		"If specified then developers mode will be used. "+
			"OAuth token will be taken from "+envTokenName+" environment variable.")
	configFile := pflag.String("config", "",
		"The controller will load its initial configuration from this file. "+
			"Omit this flag to use the default configuration values. "+
			"Command-line flags override configuration from this file.")

	logopts := log.Bind(pflag.CommandLine)

	pflag.Parse()

	ctrl.SetLogger(logopts.ConfigureLogger())

	var err error
	ctrlConfig := commonv1.ControllerConfig{}
	options := ctrl.Options{Scheme: scheme.Scheme}
	if *configFile != "" {
		options, err = options.AndFrom(ctrl.ConfigFile().AtPath(*configFile).OfKind(&ctrlConfig))
		if err != nil {
			setupLog.Error(err, "unable to load config file")
			os.Exit(1)
		}
	}

	mgr, err := ctrl.NewManager(ctrl.GetConfigOrDie(), options)

	if err != nil {
		setupLog.Error(err, "unable to start manager", "config", ctrl.GetConfigOrDie())
		os.Exit(1)
	}
	var OAuthToken = ""
	if *devMode {
		OAuthToken = os.Getenv(envTokenName)
		if OAuthToken == "" {
			setupLog.Error(err, "unable to get env variable", "envName", envTokenName)
			os.Exit(1)
		}
	}
	ctrlConfig.Options.DevMode = *devMode
	ypclient, err := yp.NewClient(
		ctrlConfig.Options.YpClusterName,
		yp.WithSystemAuthToken(),
		// orm.WithLogger(log), // TODO(torkve) need to adapt logr to arcadian logger somehow
	)
	if err != nil {
		setupLog.Error(err, "unable to create yp client")
		os.Exit(1)
	}
	defer ypclient.Close()

	tvmSecret := os.Getenv("TVM_SECRET")
	if tvmSecret == "" {
		setupLog.Error(err, "unable to get TVM_SECRET env variable")
		os.Exit(1)
	}
	tvmSettings := tvmauth.TvmAPISettings{
		SelfID: tvm.ClientID(ctrlConfig.Options.TvmID),
		ServiceTicketOptions: tvmauth.NewAliasesOptions(
			tvmSecret,
			map[string]tvm.ClientID{
				"yav": tvm.ClientID(ctrlConfig.Options.YavTvmID),
			},
		),
	}
	tvmc, err := tvmauth.NewAPIClient(tvmSettings, &nop.Logger{})
	if err != nil {
		setupLog.Error(err, "unable to create tvm client")
		os.Exit(1)
	}

	yavc, err := httpyav.NewClient()
	if err != nil {
		setupLog.Error(err, "unable to create yav client")
		os.Exit(1)
	}
	reconciler := controllers.DeployReconciler{
		Client:          mgr.GetClient(),
		Stats:           controllers.MakeStats("stage"),
		Options:         ctrlConfig.Options,
		DefaultYPClient: ypclient,
		TvmClient:       tvmc,
		YavClient:       yavc,
	}
	stageReconciler := &controllers.DeployStageReconciler{DeployReconciler: reconciler}

	abcToken := os.Getenv("ABC_TOKEN")
	if abcToken == "" {
		setupLog.Error(err, "unable to get ABC_TOKEN env variable")
		os.Exit(1)
	}
	abcClient := abc.NewClient(abcToken)

	projectReconciler := &controllers.DeployProjectReconciler{
		DeployReconciler: reconciler,
		ABCClient:        abcClient,
	}

	if err = mgr.AddMetricsExtraHandler(ctrlConfig.Unistat.Path, stageReconciler.Stats); err != nil {
		setupLog.Error(err, "unable to setup unistat", "controller", "DeployStage")
		os.Exit(1)
	}

	if err = stageReconciler.SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "DeployStage")
		os.Exit(1)
	}

	if err = projectReconciler.SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "DeployProject")
		os.Exit(1)
	}

	if err := mgr.AddHealthzCheck("healthz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up health check")
		os.Exit(1)
	}
	if err := mgr.AddReadyzCheck("readyz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up ready check")
		os.Exit(1)
	}

	setupLog.Info("starting manager")
	if err := mgr.Start(ctrl.SetupSignalHandler()); err != nil {
		setupLog.Error(err, "problem running manager")
		os.Exit(1)
	}
}
