package internal

import (
	"context"
	"errors"
	"fmt"
	"strings"
	"time"

	"github.com/go-logr/logr"
	"google.golang.org/protobuf/types/known/timestamppb"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"

	commonv1proto "a.yandex-team.ru/infra/infractl/controllers/deploy/api/common/proto_v1"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

const (
	// Failed reasons
	PreUpdateValidationFailedReason = "PreUpdateValidationFailed"
	CreationFailedReason            = "CreationFailed"
	UpdateFailedReason              = "UpdateFailed"
	MakeObjectFailedReason          = "MakeObjectFailed"
	FetchStatusFromYPFailedReason   = "FetchStatusFromYPFailed"
	FetchObjectFromYPFailedReason   = "FetchObjectFromYPFailed"
	StatusUpdateFailedReason        = "StatusUpdateFailed"
	NonMatchingFqidsReason          = "NonMatchingFqids"
	ProcessObjectFailedReason       = "ProcessObjectFailed"

	// Erros messages
	NonMatchingFqidsErrorMessage = "fqid in k8s does not match fqid in YP"

	BeforeDeleteFinalizer = "k.yandex-team.ru/deployctl-remove-owned"
)

var ErrNonMatchingFqids = errors.New(NonMatchingFqidsErrorMessage)

type ReasonedError struct {
	Err    error
	Reason string
}

func (e *ReasonedError) Error() string {
	return e.Err.Error()
}

func NewReasonedError(err error, reason string) *ReasonedError {
	return &ReasonedError{Err: err, Reason: reason}
}

func GetReasonedErrorReason(err error) string {
	reason := ""
	if reasonedErr, ok := err.(*ReasonedError); ok {
		reason = reasonedErr.Reason
	}
	return reason
}

type Reconciler struct {
	TemplateURL    string
	StatusWriter   client.StatusWriter
	UnistatUpdater interfaces.UnistatUpdater
	Client         client.Client
}

func setSyncError(
	log logr.Logger,
	syncStatus *commonv1proto.SyncStatus,
	old *commonv1proto.SyncStatus,
	e error,
) {
	reason := GetReasonedErrorReason(e)
	log.Error(e, reason)
	syncStatus.Error = &ypapi.TCondition{
		Status:  ypapi.EConditionStatus_CS_TRUE,
		Reason:  reason,
		Message: e.Error(),
	}
	if old.GetError().GetStatus() == ypapi.EConditionStatus_CS_TRUE {
		syncStatus.Error.LastTransitionTime = old.GetError().GetLastTransitionTime()
	} else {
		syncStatus.Error.LastTransitionTime = timestamppb.Now()
	}
	syncStatus.Success = nil
}

func setSyncSuccess(
	log logr.Logger,
	syncStatus *commonv1proto.SyncStatus,
	old *commonv1proto.SyncStatus,
) {
	log.V(3).Info("Saving sync success")
	syncStatus.Error = nil
	syncStatus.Success = &ypapi.TCondition{
		Status: ypapi.EConditionStatus_CS_TRUE,
		Reason: "Synced",
	}
	if old.GetSuccess().GetStatus() == ypapi.EConditionStatus_CS_TRUE {
		syncStatus.Success.LastTransitionTime = old.GetSuccess().GetLastTransitionTime()
	} else {
		syncStatus.Success.LastTransitionTime = timestamppb.Now()
	}
}

func isUpdateNeeded(log logr.Logger, kObj interfaces.KubernetesObject, curDObj interfaces.DeployObject) bool {
	annotation, err := getInfractlAnnotation(curDObj)
	if err != nil {
		// ignore error, just overwrite spec in YP
		log.Error(err, "failed to check if spec in YP is fresh enough, will update object in YP")
		return true
	}
	if kObj.SpecNewer(annotation.Generation) {
		return true
	}
	// Do nothing if k8s generation is not greater than generation stored in YP annotations.
	// So if user update YP spec bypassing controller the generation in YP annotations not changed
	// and we preserve user changes in YP spec. But as soon as user updates k8s object then generation in k8s
	// incremented and becomes greater than in YP annotations and we update spec in YP.
	// It seems that this is exactly what the user wants.
	log.Info("Spec in YP is fresh enough or was overwritten by user, skip update")
	return false
}

func addFinalizerIfNeeded(kObj interfaces.KubernetesObject) {
	if kObj.GetDeletionTimestamp().IsZero() {
		controllerutil.AddFinalizer(kObj, BeforeDeleteFinalizer)
	}
}

func (r *Reconciler) createOrUpdateObjectIfNeeded(
	ctx context.Context,
	log logr.Logger,
	deployClient DeployClient,
	kObj interfaces.KubernetesObject,
	prevDObj interfaces.DeployObject,
	maker interfaces.DeployObjectMaker,
) (interfaces.DeployObject, error) {
	oType := strings.ToLower(kObj.GetReadableObjectType())
	if prevDObj == nil {
		if len(kObj.GetFqid()) > 0 {
			// fqid may be present in k8s annotations in 2 cases:
			// 1. if we died on previous iteration before committing transaction to YP
			// 2. if user gave it to us but such object does not exist in YP
			//
			// In both cases we will rewrite this fqid by fqid of newly create YP-object
			//
			// May be in case (2) it's not so safe to just ignore this annotation, but we
			// didn't imagine this case, so ignore it for now.
			log.Info("Object does not exist, but fqid is present in annotations, ignoring fqid and overwriting it")
		}
		log.Info("Object needs creation", "object_type", oType)
		dObj, err := maker.Make(kObj, nil)
		if err != nil {
			return nil, NewReasonedError(err, MakeObjectFailedReason)
		}
		addFinalizerIfNeeded(kObj)
		if err := r.Client.Update(ctx, kObj); err != nil {
			return nil, err
		}
		commitTimestamp, err := deployClient.Create(ctx, kObj, dObj)
		if err != nil {
			return nil, NewReasonedError(err, CreationFailedReason)
		}
		return deployClient.Fetch(ctx, commitTimestamp, kObj)
	}
	log.V(3).Info(fmt.Sprintf("Fetched YP %s", oType))
	if prevDObj.GetFqid() != kObj.GetFqid() {
		return nil, NewReasonedError(ErrNonMatchingFqids, NonMatchingFqidsReason)
	}
	if !isUpdateNeeded(log, kObj, prevDObj) {
		// Push to YP is not needed (because spec is fresh enough or was overwritten
		// by user or because of some other reason), it's OK, let's just update our status in k8s
		return prevDObj, nil
	}

	if msg, err := kObj.ValidateDeployObject(prevDObj); err != nil {
		return nil, NewReasonedError(fmt.Errorf("%s: %w", msg, err), PreUpdateValidationFailedReason)
	}

	newDObj, err := maker.Make(kObj, prevDObj)
	if err != nil {
		return nil, NewReasonedError(err, MakeObjectFailedReason)
	}

	commitTimestamp, err := deployClient.Update(ctx, kObj, newDObj, prevDObj.GetYPSpecTimestamp())
	if err != nil {
		return nil, NewReasonedError(err, UpdateFailedReason)
	}
	dObj, err := deployClient.Fetch(ctx, commitTimestamp, kObj)
	if err != nil {
		return nil, NewReasonedError(err, FetchStatusFromYPFailedReason)
	}
	return dObj, nil
}

func (r *Reconciler) ProcessObject(
	ctx context.Context,
	log logr.Logger,
	deployClient DeployClient,
	dObjMaker interfaces.DeployObjectMaker,
	kObj interfaces.KubernetesObject,
) error {
	dObj, err := deployClient.Fetch(ctx, 0, kObj)
	if err != nil {
		return NewReasonedError(err, FetchObjectFromYPFailedReason)
	}
	dObj, err = r.createOrUpdateObjectIfNeeded(ctx, log, deployClient, kObj, dObj, dObjMaker)
	if err != nil {
		return err
	}
	if err = kObj.SetStatus(dObj); err != nil {
		return NewReasonedError(err, StatusUpdateFailedReason)
	}
	return nil
}

func SpecNewer(kObj interfaces.KubernetesObject, generation int64) bool {
	return kObj.GetGeneration() > generation
}

func (r *Reconciler) UpdateStatus(
	ctx context.Context,
	log logr.Logger,
	oldKObj, newKObj interfaces.KubernetesObject,
	err error,
) error {
	ss := newKObj.GetSyncStatus()
	ss.AppliedGeneration = newKObj.GetGeneration()
	ss.Url = newKObj.MakeURL(r.TemplateURL)
	if err != nil {
		setSyncError(log, ss, oldKObj.GetSyncStatus(), err)
		r.UnistatUpdater.UpdateSyncErrors(1)
		// we always return requeue-result, so we do not need additional controller logic
		// to trigger on error and requeue task immediately
	} else {
		setSyncSuccess(log, ss, oldKObj.GetSyncStatus())
	}

	equal, err := oldKObj.IsStatusEqualTo(newKObj)
	if err != nil || !equal {
		if err != nil {
			log.Error(err, "Failed to compare new and old statuses, consider status differs")
		}
		log.Info("Status differs")

		if err = r.StatusWriter.Update(ctx, newKObj); err != nil {
			return err
		}
	}
	return nil
}

func (r *Reconciler) Reconcile(
	ctx context.Context,
	log logr.Logger,
	deployClient DeployClient,
	kObj interfaces.KubernetesObject,
	maker interfaces.DeployObjectMaker,
) interfaces.KubernetesObject {
	defer unistat.MeasureMicrosecondsSince(r.UnistatUpdater.GetSyncLatency(), time.Now())
	defer r.UnistatUpdater.UpdateSyncCount(1)
	log = log.WithValues("generation", kObj.GetGeneration())
	kObjCopy, ok := kObj.DeepCopyObject().(interfaces.KubernetesObject)
	if !ok {
		msg := "failed to copy k8s object"
		log.Error(fmt.Errorf(msg), msg)
		return nil
	}

	err := r.ProcessObject(ctx, log, deployClient, maker, kObjCopy)
	if err = r.UpdateStatus(ctx, log, kObj, kObjCopy, err); err != nil {
		log.Error(err, "Failed to update status")
	}
	return kObjCopy
}
