package internal_test

import (
	"context"
	"errors"
	"testing"

	"github.com/stretchr/testify/assert"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/client/fake"
	"sigs.k8s.io/controller-runtime/pkg/log"

	projectv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/v1"
	"a.yandex-team.ru/infra/infractl/controllers/deploy/ypproject"
	"a.yandex-team.ru/infra/infractl/controllers/internal"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
)

type FakeDeployClient struct {
	ypClient     *yp.Client
	userYPClient *yp.Client
	specWriter   client.Writer
	FetchCalled  bool
	UpdateCalled bool
	CreateCalled bool

	fetchResults       []string
	fetchResultsOffset int
}

func (c *FakeDeployClient) SetFetchResults(results []string) {
	c.fetchResults = results
	c.fetchResultsOffset = 0
}

func (c *FakeDeployClient) SetBooleanFileds(create bool, fetch bool, update bool) {
	c.CreateCalled = create
	c.FetchCalled = fetch
	c.UpdateCalled = update
}

const (
	NonExistingObjectMsg = "no such object"
)

var ErrNonExistingObject = errors.New(NonExistingObjectMsg)

func (c *FakeDeployClient) Fetch(
	ctx context.Context,
	timestamp uint64,
	kObj interfaces.KubernetesObject,
) (interfaces.DeployObject, error) {
	c.SetBooleanFileds(c.CreateCalled, true, c.UpdateCalled)
	key := "nil"
	if len(c.fetchResults)-1 > c.fetchResultsOffset {
		key = c.fetchResults[c.fetchResultsOffset]
		c.fetchResultsOffset += 1
	}
	if key == "nil" {
		return nil, ErrNonExistingObject
	}
	if key == "empty" {
		return nil, nil
	}
	var dObj = internal.DeployObjectWrapper{ypproject.NewYPProject(&ypapi.TProject{}, 0), key}
	return dObj, nil
}

func (c *FakeDeployClient) Create(
	ctx context.Context,
	kObj interfaces.KubernetesObject,
	dObj interfaces.DeployObject,
) (uint64, error) {
	c.SetBooleanFileds(true, c.FetchCalled, c.UpdateCalled)
	return 0, nil
}

func (c *FakeDeployClient) Update(
	ctx context.Context,
	kObj interfaces.KubernetesObject,
	dObj interfaces.DeployObject,
	curYPSpecTimestamp uint64,
) (uint64, error) {
	c.SetBooleanFileds(c.CreateCalled, c.FetchCalled, true)
	return 0, nil
}

type FakeDeployObjectMaker struct {
}

func (f *FakeDeployObjectMaker) Make(kObj interfaces.KubernetesObject, prevDobj interfaces.DeployObject) (interfaces.DeployObject, error) {
	return ypproject.NewYPProject(&ypapi.TProject{}, 0), nil
}

type FakeUnistatUpdater struct {
}

func (f *FakeUnistatUpdater) GetSyncLatency() *unistat.Histogram {
	return &unistat.Histogram{}
}
func (f *FakeUnistatUpdater) UpdateSyncErrors(value float64) {
}
func (f *FakeUnistatUpdater) UpdateSyncCount(value float64) {
}

func getReasonedErrorObj(err error) error {
	if err == nil {
		return nil
	}
	if reasonedErr, ok := err.(*internal.ReasonedError); ok {
		return reasonedErr.Err
	}
	return nil
}

type fakeClient struct {
	client.WithWatch
}

func (c *fakeClient) Update(ctx context.Context, obj client.Object, opts ...client.UpdateOption) error {
	return nil
}
func TestProcessObject(t *testing.T) {
	var stageName1 = "stage|hash1"
	var stageName2 = "stage|hash2"
	var baseCtx = context.Background()
	log := log.FromContext(baseCtx)
	var kObj = &projectv1.DeployProject{}
	var dObjMaker = &FakeDeployObjectMaker{}
	var client = &fakeClient{fake.NewClientBuilder().Build()}
	testCases := []struct {
		name                                    string
		fetchResults                            []string
		fetchCalled, createCalled, updateCalled bool
		kuberStageName, errMsg                  string
		err                                     error
	}{
		{
			"Case when in yp there is no such object, but in kubernetes may be it exists",
			[]string{"empty"},
			true, false, false,
			stageName1, internal.FetchObjectFromYPFailedReason,
			ErrNonExistingObject,
		},
		{
			"Case when in yp such object exists, but fqid doesn't match with kuber's fqid",
			[]string{stageName1, stageName1},
			true, false, false,
			stageName2, internal.NonMatchingFqidsReason,
			internal.ErrNonMatchingFqids,
		},
		{
			"Case when in kubernetes there is no such object",
			[]string{"empty", stageName1},
			true, true, false,
			stageName1, "",
			nil,
		},
		{
			"Case when in yp such object exists and fqid matches with kuber's object",
			[]string{stageName1, stageName1, stageName1},
			true, false, true,
			stageName1, "",
			nil,
		},
	}
	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			reconciler := internal.Reconciler{
				TemplateURL:    "https://deploy.yandex-team.ru/%s/%s",
				StatusWriter:   nil,
				UnistatUpdater: nil,
				Client:         client,
			}
			var deployClient = &FakeDeployClient{}
			kObj.SetFqid(tc.kuberStageName)
			deployClient.SetBooleanFileds(false, false, false)
			deployClient.SetFetchResults(tc.fetchResults)

			var err = reconciler.ProcessObject(baseCtx, log, deployClient, dObjMaker, kObj)

			assert.Equal(t, tc.errMsg, internal.GetReasonedErrorReason(err))
			assert.Equal(t, true, errors.Is(tc.err, getReasonedErrorObj(err)))
			assert.Equal(t, tc.fetchCalled, deployClient.FetchCalled)
			assert.Equal(t, tc.createCalled, deployClient.CreateCalled)
			assert.Equal(t, tc.updateCalled, deployClient.UpdateCalled)
		})
	}
}

func TestReconcile(t *testing.T) {
	var stageName1 = "stage|hash1"
	var stageName2 = "stage|hash2"
	var baseCtx = context.Background()
	var log = log.FromContext(baseCtx)
	var kObj = &projectv1.DeployProject{}
	var deployClient = &FakeDeployClient{}
	var unistatUpdater = &FakeUnistatUpdater{}
	var fakeStatusWriter = fake.NewClientBuilder().Build().Status()
	var maker = &FakeDeployObjectMaker{}
	var client = &fakeClient{fake.NewClientBuilder().Build()}
	kObj.SetFqid(stageName1)
	testCases := []struct {
		name, domain, correctURL string
		fetchResults             []string
		checkSuccess             bool
	}{
		{
			"Case when sync status was set to success",
			"https://man-pre.deploy.yandex-team.ru", "https://man-pre.deploy.yandex-team.ru/projects/",
			[]string{stageName1, stageName1, stageName1},
			true,
		},
		{
			"Case when sync status was set to error, because in kubernetes there are no such object",
			"https://test.deploy.yandex-team.ru", "https://test.deploy.yandex-team.ru/projects/",
			[]string{stageName1},
			false,
		},
		{
			"Case when sync status was set to error, because yp's and kubernetes object have different fqid",
			"https://deploy.yandex-team.ru", "https://deploy.yandex-team.ru/projects/",
			[]string{stageName2, stageName2},
			false,
		},
	}
	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			deployClient.SetFetchResults(tc.fetchResults)
			reconciler := internal.Reconciler{
				TemplateURL:    tc.domain + "/%s/%s",
				StatusWriter:   fakeStatusWriter,
				UnistatUpdater: unistatUpdater,
				Client:         client,
			}
			var newkObj = reconciler.Reconcile(baseCtx, log, deployClient, kObj, maker)
			assert.NotEqual(t, nil, newkObj)
			var syncStatus = newkObj.GetSyncStatus()
			if tc.checkSuccess {
				assert.NotEqual(t, nil, syncStatus.Success)
				assert.Equal(t, ypapi.EConditionStatus_CS_TRUE, syncStatus.GetSuccess().GetStatus())
			} else {
				assert.NotEqual(t, nil, syncStatus.Error)
				assert.Equal(t, ypapi.EConditionStatus_CS_TRUE, syncStatus.GetError().GetStatus())
			}
			assert.Equal(t, tc.correctURL, syncStatus.Url)
		})
	}
}
