package v1

import (
	"fmt"

	"google.golang.org/protobuf/proto"

	commonv1proto "a.yandex-team.ru/infra/infractl/controllers/deploy/api/common/proto_v1"
	ctrls "a.yandex-team.ru/infra/infractl/controllers/internal"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yp/go/yp"
)

const (
	ProjectFqidAnnotation = "nanny.yandex-team.ru/fqid"
)

func (o *NannyRuntime) GetObjectType() yp.ObjectType {
	return yp.ObjectTypeMultiClusterReplicaSet
}

func (o *NannyRuntime) GetReadableObjectType() string {
	return "MultiClusterReplicaSet"
}

func (o *NannyRuntime) GetAccountID() string {
	return o.Spec.AccountId
}

func (o *NannyRuntime) GetSyncStatus() *commonv1proto.SyncStatus {
	if o.Status.GetSync() == nil {
		o.Status.Sync = &commonv1proto.SyncStatus{}
	}
	return o.Status.GetSync()
}

func (o *NannyRuntime) IsStatusEqualTo(kObj interfaces.KubernetesObject) (bool, error) {
	var equal bool
	kProject, ok := kObj.(*NannyRuntime)
	if !ok {
		return equal, fmt.Errorf("failed to cast object to k8s project")
	}
	return proto.Equal(kProject.Status, o.Status), nil
}

func (o *NannyRuntime) SpecNewer(generation int64) bool {
	return ctrls.SpecNewer(o, generation)
}

func (o *NannyRuntime) MakeURL(template string) string {
	return fmt.Sprintf(template, "multicluster-replica-sets", o.GetName())
}

func (o *NannyRuntime) FillDeployObject(response *yp.GetObjectResponse) (interfaces.DeployObject, error) {
	mcrs := &ypapi.TMultiClusterReplicaSet{}
	if err := response.Fill(mcrs); err != nil {
		return nil, err
	}

	dObj := NewYPMultiClusterReplicaSet(mcrs, response.Timestamps()[1])
	return dObj, nil
}

func (o *NannyRuntime) SetStatus(dObj interfaces.DeployObject) error {
	mcrs, ok := dObj.GetYPObject().(*ypapi.TMultiClusterReplicaSet)
	if !ok {
		return fmt.Errorf("failed to cast deploy project to YP project")
	}
	o.Status.McrsStatus = mcrs.GetStatus().DeepCopy()
	return nil
}

func (o *NannyRuntime) ValidateDeployObject(_ interfaces.DeployObject) (string, error) {
	return "", nil
}

func (o *NannyRuntime) SetFqid(fqid string) {
	if o.Annotations == nil {
		o.Annotations = make(map[string]string)
	}
	o.Annotations[ProjectFqidAnnotation] = fqid
}

func (o *NannyRuntime) GetFqid() string {
	return o.GetAnnotations()[ProjectFqidAnnotation]
}
