package v1

import (
	"fmt"

	"a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/proto/core/ytree"
	"a.yandex-team.ru/yt/go/yson"
)

const (
	MainDiskVolumeRequestID = "main-disk"
	MainBoxID               = "box"
	NetworkMacroSearchSand  = "_SEARCHSAND_"
)

func makePodAgentPayloadMeta(runtimeVersion string) (*ypapi.TPodSpec_TPodAgentDeploymentMeta, error) {
	if runtimeVersion == "v1" || runtimeVersion == "" {
		url := "rbtorrent:13f5ac741e8f2b5afd34b6a54402459458e2f331"
		checksum := "MD5:013e2176e1a8d52f69a7a93dc13c02f0"
		layerURL := "rbtorrent:4b8ccda19a8b6927f4dcc5d5eb019efca6c753c7"
		layerChecksum := "MD5:cc3e23f9a22b3202aa3613679c1bb6c5"
		return &ypapi.TPodSpec_TPodAgentDeploymentMeta{
			Url:      &url,
			Checksum: &checksum,
			Layers: []*ypapi.TPodSpec_TPodAgentDeploymentMeta_TLayer{
				{
					Url:      &layerURL,
					Checksum: &layerChecksum,
				},
			},
		}, nil
	} else if runtimeVersion == "v2" {
		url := "rbtorrent:d068d90d38f859dbb5195b3fc4dec6c0554826ae"
		checksum := "MD5:2c729a19b0c3aab342f3fd3ce24eb0ad"
		binaryRevision := uint64(3214891279)
		layerChecksum := "MD5:cc3e23f9a22b3202aa3613679c1bb6c5"
		layerURL := "rbtorrent:4b8ccda19a8b6927f4dcc5d5eb019efca6c753c7"

		return &ypapi.TPodSpec_TPodAgentDeploymentMeta{
			Url:            &url,
			Checksum:       &checksum,
			BinaryRevision: &binaryRevision,
			Configuration: map[string]string{
				"PodAgentMode": "BOX_MODE",
			},
			Layers: []*ypapi.TPodSpec_TPodAgentDeploymentMeta_TLayer{
				{
					Url:      &layerURL,
					Checksum: &layerChecksum,
				},
			},
		}, nil
	} else {
		return nil, fmt.Errorf("unsupported runtime version: %s", runtimeVersion)
	}
}

func makeDiskVolumeRequests() []*ypapi.TPodSpec_TDiskVolumeRequest {
	diskVolumeRequests := make([]*ypapi.TPodSpec_TDiskVolumeRequest, 0)
	stroageClass := "hdd"
	makeLayerID := MainDiskVolumeRequestID
	capacity := uint64(3221225472)
	bandwidthLimit := uint64(31457280)
	bandwidthGuarantee := uint64(15728640)
	diskVolumeRequests = append(diskVolumeRequests, &ypapi.TPodSpec_TDiskVolumeRequest{
		Id:           &makeLayerID,
		StorageClass: &stroageClass,
		ConcretePolicy: &ypapi.TPodSpec_TDiskVolumeRequest_QuotaPolicy{
			QuotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
				Capacity:           &capacity,
				BandwidthLimit:     &bandwidthLimit,
				BandwidthGuarantee: &bandwidthGuarantee,
			},
		},
	})
	capacityInfra := uint64(1073741824)
	idInfra := "pod_agent"
	keyInfra := "used_by_infra"
	ysonTrue, _ := yson.Marshal(true) // id-todo: move to yputils
	diskVolumeRequests = append(diskVolumeRequests, &ypapi.TPodSpec_TDiskVolumeRequest{
		Id:           &idInfra,
		StorageClass: &stroageClass,
		Labels: &ytree.TAttributeDictionary{
			Attributes: []*ytree.TAttribute{
				{
					Key:   &keyInfra,
					Value: ysonTrue,
				},
			},
		},
		ConcretePolicy: &ypapi.TPodSpec_TDiskVolumeRequest_QuotaPolicy{
			QuotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
				Capacity: &capacityInfra,
			},
		},
	})
	return diskVolumeRequests
}

func makeLayerID(i int) string {
	return fmt.Sprintf("layer-%d", i)
}

func makeLayers(runtime *NannyRuntime) ([]*podagent.TLayer, error) {
	instanceType := runtime.GetSpec().GetInstanceSpec().Type
	if instanceType != nanny_repo.InstanceSpec_SANDBOX_LAYERS {
		return nil, fmt.Errorf("instances with type: %s not yes supported", instanceType.String())
	}

	layers := runtime.GetSpec().GetInstanceSpec().GetLayersConfig().GetLayer()
	podagentLayers := make([]*podagent.TLayer, 0, len(layers))
	for i, layer := range layers {
		podagentLayers = append(podagentLayers, &podagent.TLayer{
			Id:               makeLayerID(i),
			Checksum:         "EMPTY:",
			DownloadMethod:   &podagent.TLayer_Url{Url: layer.Url[0]},
			VirtualDiskIdRef: MainDiskVolumeRequestID,
		})
	}
	return podagentLayers, nil
}

func makeBoxLayerRefs(runtime *NannyRuntime) []string {
	layers := runtime.GetSpec().GetInstanceSpec().GetLayersConfig().GetLayer()
	refs := make([]string, 0, len(layers))
	for i := range layers {
		refs = append(refs, makeLayerID(i))
	}
	return refs
}

func makeBox(runtime *NannyRuntime) (*podagent.TBox, error) {
	init := make([]*podagent.TUtilityContainer, 0)
	for _, i := range runtime.GetSpec().GetInstanceSpec().GetInitContainers() {
		initContainer, err := makeContainerSpec(i)
		if err != nil {
			return nil, err
		}
		init = append(init, initContainer)
	}
	return &podagent.TBox{
		Id:               MainBoxID,
		SpecificType:     "default",
		VirtualDiskIdRef: MainDiskVolumeRequestID,
		Init:             init,
		Rootfs: &podagent.TRootfsVolume{
			LayerRefs: makeBoxLayerRefs(runtime),
		},
	}, nil
}

func makeResourceRequests() *ypapi.TPodSpec_TResourceRequests {
	MemoryLimit := uint64(1610612736)
	MemoryGuarantee := uint64(1610612736)
	AnonymousMemoryLimit := uint64(1436129690)
	VcpuLimit := uint64(500)
	VcpuGuarantee := uint64(500)
	NetworkBandwidthGuarantee := uint64(10485760)
	return &ypapi.TPodSpec_TResourceRequests{
		MemoryLimit:               &MemoryLimit,
		MemoryGuarantee:           &MemoryGuarantee,
		AnonymousMemoryLimit:      &AnonymousMemoryLimit,
		VcpuLimit:                 &VcpuLimit,
		VcpuGuarantee:             &VcpuGuarantee,
		NetworkBandwidthGuarantee: &NetworkBandwidthGuarantee,
	}
}

func makeIP6AddressRequests() []*ypapi.TPodSpec_TIP6AddressRequest {
	searchsand := NetworkMacroSearchSand
	Priority := int32(1)
	vlanIDBackbone := "backbone"
	vlanIDFastbone := "fastbone"
	enableDNS := true
	return []*ypapi.TPodSpec_TIP6AddressRequest{
		{
			NetworkId: &searchsand,
			Priority:  &Priority,
			VlanId:    &vlanIDBackbone,
			EnableDns: &enableDNS,
		},
		{
			NetworkId: &searchsand,
			VlanId:    &vlanIDFastbone,
			EnableDns: &enableDNS,
		},
	}
}

func makeIP6SubnetRequests() []*ypapi.TPodSpec_TIP6SubnetRequest {
	searchsand := NetworkMacroSearchSand
	key := "id"
	VlanID := "backbone"
	return []*ypapi.TPodSpec_TIP6SubnetRequest{
		{
			NetworkId: &searchsand,
			Labels: &ytree.TAttributeDictionary{
				Attributes: []*ytree.TAttribute{
					{
						Key:   &key,
						Value: []byte("boxes_subnet"),
					},
				},
			},
			VlanId: &VlanID,
		},
	}
}

func makePodTemplateSpec(runtime *NannyRuntime) (*ypapi.TPodTemplateSpec, error) {
	isTrue := true
	workloads, secrets, err := makeWorkloads(runtime, MainBoxID)
	if err != nil {
		return nil, err
	}
	if len(secrets) > 0 {
		return nil, fmt.Errorf("secrets is not yet supported")
	}
	workloadsRefs := make([]*podagent.TMutableWorkload, 0, len(workloads))
	for _, w := range workloads {
		workloadsRefs = append(workloadsRefs, &podagent.TMutableWorkload{
			WorkloadRef: w.Id,
		})
	}
	podAgentPayloadMeta, err := makePodAgentPayloadMeta(runtime.Spec.RuntimeVersion)
	if err != nil {
		return nil, err
	}
	box, err := makeBox(runtime)
	if err != nil {
		return nil, err
	}

	layers, err := makeLayers(runtime)
	if err != nil {
		return nil, err
	}

	return &ypapi.TPodTemplateSpec{
		Spec: &ypapi.TPodSpec{
			NetworkSettings: &ypapi.TPodSpec_TNetworkSettings{
				IpLimit: &isTrue,
			},
			DiskVolumeRequests: makeDiskVolumeRequests(),
			PodAgentPayload: &ypapi.TPodSpec_TPodAgentPayload{
				Meta: podAgentPayloadMeta,
				Spec: &podagent.TPodAgentSpec{
					Boxes:            []*podagent.TBox{box},
					Workloads:        workloads,
					MutableWorkloads: workloadsRefs,
					Resources: &podagent.TResourceGang{
						Layers: layers,
					},
				},
			},
			Ip6AddressRequests: makeIP6AddressRequests(),
			Ip6SubnetRequests:  makeIP6SubnetRequests(),
			ResourceRequests:   makeResourceRequests(),
		},
	}, nil
}

func makeDeploymentStrategy() (*ypapi.TMultiClusterReplicaSetSpec_TDeploymentStrategy, error) {
	return &ypapi.TMultiClusterReplicaSetSpec_TDeploymentStrategy{
		MaxUnavailable:              1,
		MaxTolerableDowntimeSeconds: 3600,
		MaxTolerableDowntimePods:    1,
	}, nil
}

func makeClusters() ([]*ypapi.TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences, error) {
	return []*ypapi.TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences{
		{
			Cluster: "man-pre",
			Spec: &ypapi.TMultiClusterReplicaSetSpec_TReplicaSetSpecPreferences{
				ReplicaCount: 1,
			},
		},
	}, nil
}

func MakeMultiClusterReplicaSet(runtime *NannyRuntime) (mcrs *ypapi.TMultiClusterReplicaSet, err error) {
	podTemplateSpec, err := makePodTemplateSpec(runtime)
	if err != nil {
		return nil, err
	}
	deploymentStrategy, err := makeDeploymentStrategy()
	if err != nil {
		return nil, err
	}
	clusters, err := makeClusters()
	if err != nil {
		return nil, err
	}
	mcrs = &ypapi.TMultiClusterReplicaSet{
		Spec: &ypapi.TMultiClusterReplicaSetSpec{
			PodTemplateSpec:    podTemplateSpec,
			DeploymentStrategy: deploymentStrategy,
			Clusters:           clusters,
			AccountId:          runtime.Spec.AccountId,
		},
	}
	return
}
