package v1

import (
	"fmt"
	"sort"
	"strconv"
	"strings"

	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

func makeContainerSpec(container *nanny_repo.Container) (*podagent.TUtilityContainer, error) {
	vcpuLimit := uint64(0)
	memoryLimit := uint64(0)
	if container.GetResourceRequest() != nil {
		for _, r := range container.GetResourceRequest().GetLimit() {
			if r.Name == "mem" && r.Type == nanny_repo.ComputeResource_SCALAR {
				memoryLimit = uint64(r.Scalar.Value)
			} else if r.Name == "cpu" && r.Type == nanny_repo.ComputeResource_SCALAR {
				vcpuLimit = uint64(r.Scalar.Value)
			} else {
				return nil, fmt.Errorf("container '%s' has unknown resource limit '%s' with type %s",
					container.Name, r.Name, r.Type.String())
			}
		}
	}
	commandLine, err := makeCommandLine(container.GetCommand())
	if err != nil {
		return nil, err
	}
	c := &podagent.TUtilityContainer{
		ComputeResources: &podagent.TComputeResources{
			VcpuLimit:       vcpuLimit,
			VcpuGuarantee:   vcpuLimit,
			MemoryLimit:     memoryLimit,
			MemoryGuarantee: memoryLimit,
		},
		CommandLine: commandLine,
		// Cwd:      "", // id-todo: how nanny change cwd?
	}
	return c, nil
}

func makeCommandLine(command []string) (string, error) {
	// id-todo: maybe repalace with https://a.yandex-team.ru/arcadia/infra/nanny/instancectl/src/instancectl/lib/process/porto_container.py?rev=r9708192#L20-33
	if len(command) == 0 {
		return "", fmt.Errorf("command is empty")
	}
	var sb []string
	needQuote := false
	for _, s := range command {
		needQuote = len(s) == 0 ||
			strings.Contains(s, " ") ||
			strings.Contains(s, "\t") ||
			strings.Contains(s, "\\") ||
			strings.Contains(s, "\"") ||
			strings.Contains(s, "'")
		if needQuote {
			sb = append(sb, fmt.Sprintf("%q", s))
		} else {
			sb = append(sb, s)
		}
	}
	return strings.Join(sb, " "), nil
}

func makePort(portValue string) uint32 {
	port, err := strconv.Atoi(portValue)
	if err != nil {
		port = 80 // если не указан порт, то берется дефолтный
	}
	return uint32(port)
}

func makeHTTPGetCheck(nannyHTTPGetCheck *nanny_repo.HTTPGetAction, timeLimit *podagent.TTimeLimit) *podagent.THttpGet {
	// id-todo: В логике nanny успешным считается ответ с кодом < 400, надо проверить что в deploy так же
	return &podagent.THttpGet{
		Port:      makePort(nannyHTTPGetCheck.GetPort()),
		Path:      nannyHTTPGetCheck.GetPath(),
		TimeLimit: timeLimit,
		Answer: &podagent.THttpGet_Any{
			// Если выставлено в true, ответ сервера игнорируется (проверяется только код ответа)
			Any: true,
		},
	}
}

func makeTCPCheck(nannyTCPCheck *nanny_repo.TCPSocketAction, timeLimit *podagent.TTimeLimit) *podagent.TTcpCheck {
	return &podagent.TTcpCheck{
		Port:      makePort(nannyTCPCheck.GetPort()),
		TimeLimit: timeLimit,
	}
}

func makeReadinessCheck(readinessProbe *nanny_repo.Probe) (*podagent.TReadinessCheck, error) {
	handlers := readinessProbe.GetHandlers()
	if len(handlers) == 0 {
		return &podagent.TReadinessCheck{
			Backend:          &podagent.TReadinessCheck_StartContainerAliveCheck{},
			FailureThreshold: uint32(readinessProbe.GetFailureThreshold()),
			SuccessThreshold: uint32(readinessProbe.GetSuccessThreshold()),
		}, nil
	}
	if len(handlers) > 1 {
		return nil, fmt.Errorf("at most one readiness probe is supported")
	}
	h := handlers[0]
	timeLimit := &podagent.TTimeLimit{
		InitialDelayMs:       uint64(readinessProbe.GetInitialDelaySeconds() * 1000),
		MaxRestartPeriodMs:   uint64(readinessProbe.GetMaxPeriodSeconds() * 1000),
		MinRestartPeriodMs:   uint64(readinessProbe.GetMinPeriodSeconds() * 1000),
		RestartPeriodBackOff: uint64(readinessProbe.GetPeriodBackoff() * 1000),
		MaxExecutionTimeMs:   30 * 60 * 1000,
	}
	switch h.Type {
	case nanny_repo.Handler_EXEC:
		commandLine, err := makeCommandLine(h.GetExecAction().GetCommand())
		if err != nil {
			return nil, err
		}
		return &podagent.TReadinessCheck{
			Backend: &podagent.TReadinessCheck_Container{
				Container: &podagent.TUtilityContainer{
					CommandLine: commandLine,
					TimeLimit:   timeLimit,
				},
			},
			FailureThreshold: uint32(readinessProbe.GetFailureThreshold()),
			SuccessThreshold: uint32(readinessProbe.GetSuccessThreshold()),
		}, nil
	case nanny_repo.Handler_HTTP_GET:
		return &podagent.TReadinessCheck{
			Backend: &podagent.TReadinessCheck_HttpGet{
				HttpGet: makeHTTPGetCheck(h.GetHttpGet(), timeLimit),
			},
			FailureThreshold: uint32(readinessProbe.GetFailureThreshold()),
			SuccessThreshold: uint32(readinessProbe.GetSuccessThreshold()),
		}, nil
	case nanny_repo.Handler_TCP_SOCKET:
		return &podagent.TReadinessCheck{
			Backend: &podagent.TReadinessCheck_TcpCheck{
				TcpCheck: makeTCPCheck(h.GetTcpSocket(), timeLimit),
			},
			FailureThreshold: uint32(readinessProbe.GetFailureThreshold()),
			SuccessThreshold: uint32(readinessProbe.GetSuccessThreshold()),
		}, nil
	default:
		return nil, fmt.Errorf("readiness probe with type %s does not supported", h.Type.String())
	}
}

func makeStopPolicy(lifecycle *nanny_repo.Lifecycle) (*podagent.TStopPolicy, error) {
	// id-todo: lifecycle.StopGracePeriodSeconds
	// id-todo: lifecycle.TerminationGracePeriodSeconds

	timeLimit := &podagent.TTimeLimit{
		InitialDelayMs:       0,
		RestartPeriodScaleMs: 0,
		RestartPeriodBackOff: 0,
		MaxRestartPeriodMs:   3153600000000,
		MinRestartPeriodMs:   1000,
		MaxExecutionTimeMs:   30000,
	}
	h := lifecycle.GetPreStop()
	if h == nil {
		return nil, nil
	}
	switch h.Type {
	case nanny_repo.Handler_EXEC:
		commandLine, err := makeCommandLine(h.GetExecAction().GetCommand())
		if err != nil {
			return nil, err
		}
		return &podagent.TStopPolicy{
			Backend: &podagent.TStopPolicy_Container{
				Container: &podagent.TUtilityContainer{
					CommandLine: commandLine,
					TimeLimit:   timeLimit,
				},
			},
			MaxTries: 1,
		}, nil
	case nanny_repo.Handler_HTTP_GET:
		return &podagent.TStopPolicy{
			Backend: &podagent.TStopPolicy_HttpGet{
				HttpGet: makeHTTPGetCheck(h.GetHttpGet(), timeLimit),
			},
			MaxTries: 1,
		}, nil
	default:
		return nil, fmt.Errorf("prestop handler with %s type does not supported", h.Type.String())
	}
}

func parseEnvValues(
	containerEnvVars []*nanny_repo.EnvVar,
	sortByKey bool,
) ([]*podagent.TEnvVar, map[string]*ypapi.TSecretRef, error) {
	envVars := make([]*podagent.TEnvVar, 0, len(containerEnvVars))
	secretRefs := map[string]*ypapi.TSecretRef{}
	for _, env := range containerEnvVars {
		key := env.GetName()
		var envValue *podagent.TEnvVarValue
		sourceType := env.GetValueFrom().GetType()
		if sourceType == nanny_repo.EnvVarSource_LITERAL_ENV {
			envValue = &podagent.TEnvVarValue{
				Value: &podagent.TEnvVarValue_LiteralEnv{
					LiteralEnv: &podagent.LiteralEnvSelector{
						Value: env.GetValueFrom().GetLiteralEnv().GetValue(),
					},
				},
			}
		} else if sourceType == nanny_repo.EnvVarSource_VAULT_SECRET_ENV {
			sourceSecretVal := env.GetValueFrom().GetVaultSecretEnv().GetVaultSecret()
			secret := &deployutil.Secret{
				ID:      sourceSecretVal.GetSecretId(),
				Version: sourceSecretVal.GetSecretVer(),
				Key:     sourceSecretVal.GetSecretName(),
			}
			alias := secret.MakeAlias()
			envValue = &podagent.TEnvVarValue{
				Value: &podagent.TEnvVarValue_SecretEnv{
					SecretEnv: &podagent.SecretSelector{
						Alias: alias,
						Id:    secret.Key,
					},
				},
			}
			secretRefs[alias] = &ypapi.TSecretRef{
				SecretId:      &secret.ID,
				SecretVersion: &secret.Version,
			}
		} else {
			return nil, nil, fmt.Errorf("unsupported env: %s type: %s", key, sourceType.String())
		}

		envVars = append(envVars, &podagent.TEnvVar{
			Name:  key,
			Value: envValue,
		})
	}

	if sortByKey {
		sort.Slice(envVars, func(i, j int) bool {
			return envVars[i].Name < envVars[j].Name
		})
	}

	return envVars, secretRefs, nil
}

func makeWorkload(
	container *nanny_repo.Container,
	boxRef string,
) (*podagent.TWorkload, map[string]*ypapi.TSecretRef, error) {
	envVars, secretRefs, err := parseEnvValues(container.GetEnv(), true)
	if err != nil {
		return nil, nil, err
	}
	start, err := makeContainerSpec(container)
	if err != nil {
		return nil, nil, err
	}
	readness, err := makeReadinessCheck(container.GetReadinessProbe())
	if err != nil {
		return nil, nil, err
	}
	stop, err := makeStopPolicy(container.GetLifecycle())
	if err != nil {
		return nil, nil, err
	}
	return &podagent.TWorkload{
		Id:             container.Name,
		BoxRef:         boxRef,
		Env:            envVars,
		Start:          start,
		ReadinessCheck: readness,
		StopPolicy:     stop,
		TransmitLogs:   false, // id-todo: think about it
		// Init: nil, 			// init containers locate in box level
		// LivenessCheck:  nil, // id-todo: maybe make default?
		// DestroyPolicy:  nil,
	}, secretRefs, nil
}

func makeWorkloads(
	nannyRuntime *NannyRuntime,
	boxRef string,
) ([]*podagent.TWorkload, map[string]*ypapi.TSecretRef, error) {
	containers := nannyRuntime.GetSpec().GetInstanceSpec().GetContainers()
	if len(containers) == 0 {
		return nil, nil, fmt.Errorf("spec.instance_spec.containers is empty")
	}
	workloads := make([]*podagent.TWorkload, 0, len(containers))
	secretRefs := map[string]*ypapi.TSecretRef{}

	for _, c := range containers {
		cWorkload, wSecretRefs, err := makeWorkload(c, boxRef)
		if err != nil {
			return nil, nil, fmt.Errorf("translate container %s failed: %w", c.GetName(), err)
		}
		for alias, ref := range wSecretRefs {
			secretRefs[alias] = ref
		}
		workloads = append(workloads, cWorkload)
	}
	sort.Slice(workloads, func(i, j int) bool {
		return workloads[i].Id < workloads[j].Id
	})
	return workloads, secretRefs, nil
}
