/*
Copyright 2022.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package controllers

import (
	"context"
	"time"

	"google.golang.org/protobuf/proto"
	"k8s.io/apimachinery/pkg/runtime"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	k8slog "sigs.k8s.io/controller-runtime/pkg/log"

	ctrls "a.yandex-team.ru/infra/infractl/controllers/internal"
	runtimev1 "a.yandex-team.ru/infra/infractl/controllers/nanny/api/runtime/v1"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/yp/go/proto/ypapi"

	"a.yandex-team.ru/yp/go/yp"
)

func shouldIncRevision(spec *ypapi.TMultiClusterReplicaSetSpec, prevSpec *ypapi.TMultiClusterReplicaSetSpec) bool {
	if prevSpec == nil {
		return true
	}
	return !proto.Equal(prevSpec, spec)
}

type ypMultiClusterReplicaSetMaker struct{}

func (ms *ypMultiClusterReplicaSetMaker) Make(
	kObj interfaces.KubernetesObject,
	prevDobj interfaces.DeployObject,
) (interfaces.DeployObject, error) {
	dNannyRuntime := kObj.(*runtimev1.NannyRuntime)

	mcrs, err := runtimev1.MakeMultiClusterReplicaSet(dNannyRuntime)
	if err != nil {
		return nil, err
	}
	spec := mcrs.GetSpec()
	var prevSpec *ypapi.TMultiClusterReplicaSetSpec
	if prevDobj != nil {
		prevSpec = prevDobj.GetYPObject().(*ypapi.TMultiClusterReplicaSet).GetSpec()
		spec.Revision = prevSpec.GetRevision()
		spec.GetPodTemplateSpec().GetSpec().GetPodAgentPayload().GetSpec().Revision = uint32(spec.Revision)
	}

	if shouldIncRevision(spec, prevSpec) {
		spec.Revision++
		spec.GetPodTemplateSpec().GetSpec().GetPodAgentPayload().GetSpec().Revision = uint32(spec.Revision)
	}

	return runtimev1.NewYPMultiClusterReplicaSet(
		&ypapi.TMultiClusterReplicaSet{
			Meta: &ypapi.TMultiClusterReplicaSetMeta{
				Id: kObj.GetName(),
			},
			Spec: spec,
		},
		0,
	), nil
}

// NannyRuntimeReconciler reconciles a NannyRuntime object
type NannyRuntimeReconciler struct {
	client.Client
	Scheme   *runtime.Scheme
	Stats    *Stats
	YpClient *yp.Client
}

//+kubebuilder:rbac:groups=nanny.yandex-team.ru,resources=NannyRuntimes,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=nanny.yandex-team.ru,resources=NannyRuntimes/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=nanny.yandex-team.ru,resources=NannyRuntimes/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the NannyRuntime object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.11.2/pkg/reconcile
//id-xxx: Reconcile function
func (r *NannyRuntimeReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := k8slog.FromContext(ctx)
	log.Info("reconcile starting")
	defer log.Info("reconcile finishing")

	nannyRuntime := &runtimev1.NannyRuntime{}
	result = ctrl.Result{
		RequeueAfter: time.Second * 60,
	}

	if err = r.Get(ctx, req.NamespacedName, nannyRuntime); err != nil {
		log.Error(err, "unable to fetch nanny runtime")
		err = client.IgnoreNotFound(err)
		if err == nil {
			result = ctrl.Result{}
		}
		return
	}

	deployClient := dclient.NewDeployClient(r.YpClient, r.YpClient, r)

	log = log.WithValues("generation", nannyRuntime.GetGeneration())

	reconciler := ctrls.Reconciler{
		TemplateURL:    "https://yd.yandex-team.ru/yp/xdc/%s/%s",
		StatusWriter:   r.Status(),
		UnistatUpdater: r.Stats,
	}

	_ = reconciler.Reconcile(ctx, log, deployClient, nannyRuntime, &ypMultiClusterReplicaSetMaker{})
	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *NannyRuntimeReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&runtimev1.NannyRuntime{}).
		Complete(r)
}
