/*
Copyright 2022.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package main

import (
	"os"

	// Import all Kubernetes client auth plugins (e.g. Azure, GCP, OIDC, etc.)
	// to ensure that exec-entrypoint and run can make use of them.
	_ "k8s.io/client-go/plugin/pkg/client/auth"

	"github.com/spf13/pflag"
	"k8s.io/apimachinery/pkg/runtime"
	utilruntime "k8s.io/apimachinery/pkg/util/runtime"
	clientgoscheme "k8s.io/client-go/kubernetes/scheme"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/healthz"

	"a.yandex-team.ru/infra/infractl/internal/log"

	"a.yandex-team.ru/yp/go/yp"

	nannyruntimev1 "a.yandex-team.ru/infra/infractl/controllers/nanny/api/runtime/v1"
	"a.yandex-team.ru/infra/infractl/controllers/nanny/controllers"
	//+kubebuilder:scaffold:imports
)

var (
	scheme   = runtime.NewScheme()
	setupLog = ctrl.Log.WithName("setup")
)

func init() {
	utilruntime.Must(clientgoscheme.AddToScheme(scheme))

	utilruntime.Must(nannyruntimev1.AddToScheme(scheme))
	//+kubebuilder:scaffold:scheme
}

func main() {
	configFile := pflag.String("config", "",
		"The controller will load its initial configuration from this file. "+
			"Omit this flag to use the default configuration values. "+
			"Command-line flags override configuration from this file.")

	logopts := log.Bind(pflag.CommandLine)

	pflag.Parse()
	ctrl.SetLogger(logopts.ConfigureLogger())

	var err error
	ctrlConfig := controllers.ControllerConfig{}
	options := ctrl.Options{Scheme: scheme}
	if *configFile != "" {
		options, err = options.AndFrom(ctrl.ConfigFile().AtPath(*configFile).OfKind(&ctrlConfig))
		if err != nil {
			setupLog.Error(err, "unable to load config file")
			os.Exit(1)
		}
	}

	mgr, err := ctrl.NewManager(ctrl.GetConfigOrDie(), options)

	if err != nil {
		setupLog.Error(err, "unable to start manager", "config", ctrl.GetConfigOrDie())
		os.Exit(1)
	}

	ypClient, err := yp.NewClient(
		ctrlConfig.Options.YpClusterName,
		yp.WithSystemAuthToken(),
		// orm.WithLogger(log), // TODO(torkve) need to adapt logr to arcadian logger somehow
	)
	if err != nil {
		setupLog.Error(err, "unable to create yp client")
		os.Exit(1)
	}
	defer ypClient.Close()

	serviceReconciler := &controllers.NannyRuntimeReconciler{
		Client:   mgr.GetClient(),
		Scheme:   mgr.GetScheme(),
		Stats:    controllers.MakeStats("nannyruntime"),
		YpClient: ypClient,
	}

	if err = serviceReconciler.SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "NannyRuntime")
		os.Exit(1)
	}
	//+kubebuilder:scaffold:builder

	if err = mgr.AddMetricsExtraHandler(ctrlConfig.Unistat.Path, serviceReconciler.Stats); err != nil {
		setupLog.Error(err, "unable to setup unistat", "controller", "NannyRuntime")
		os.Exit(1)
	}

	if err := mgr.AddHealthzCheck("healthz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up health check")
		os.Exit(1)
	}
	if err := mgr.AddReadyzCheck("readyz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up ready check")
		os.Exit(1)
	}

	setupLog.Info("starting manager")
	if err := mgr.Start(ctrl.SetupSignalHandler()); err != nil {
		setupLog.Error(err, "problem running manager")
		os.Exit(1)
	}
}
