package controllers

import (
	"context"
	"fmt"
	"reflect"
	"strings"
	"time"

	"github.com/go-logr/logr"
	corev1 "k8s.io/api/core/v1"
	rbacv1 "k8s.io/api/rbac/v1"
	"k8s.io/apimachinery/pkg/api/errors"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/log"

	"a.yandex-team.ru/infra/infractl/controllers/roles/api/proto_v1"
	rolesv1 "a.yandex-team.ru/infra/infractl/controllers/roles/api/v1"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

const (
	RootRoleName                 = "root"
	NamespaceRootRoleBindingName = "namespace-root-binding"
	NamespaceRolesConfName       = "roles-conf"
	NamespaceEditorRoleSuffix    = "namespace-editor"
	NamespaceEditorBindingSuffix = "namespace-editor-binding"
)

func makeRolesStatusFailure(msg string) *proto_v1.Status {
	return &proto_v1.Status{
		Failure: &ypapi.TCondition{
			Status:  ypapi.EConditionStatus_CS_TRUE,
			Reason:  "",
			Message: msg,
		},
		Success: &ypapi.TCondition{
			Status:  ypapi.EConditionStatus_CS_FALSE,
			Reason:  "",
			Message: msg,
		},
	}
}

func makeRolesStatusSuccess(msg string) *proto_v1.Status {
	return &proto_v1.Status{
		Failure: &ypapi.TCondition{
			Status:  ypapi.EConditionStatus_CS_FALSE,
			Reason:  "",
			Message: msg,
		},
		Success: &ypapi.TCondition{
			Status:  ypapi.EConditionStatus_CS_TRUE,
			Reason:  "",
			Message: msg,
		},
	}
}

func makeNamespaceRootRoleBinding(nsName string, groupName string) *rbacv1.RoleBinding {
	return &rbacv1.RoleBinding{
		ObjectMeta: metav1.ObjectMeta{
			Namespace: nsName,
			Name:      NamespaceRootRoleBindingName,
		},
		RoleRef: rbacv1.RoleRef{
			Kind: "ClusterRole",
			Name: RootRoleName,
		},
		Subjects: []rbacv1.Subject{
			{
				Kind: "Group",
				Name: groupName,
			},
		},
	}
}

func makeNamespaceEditorRole(nsName string) *rbacv1.ClusterRole {
	return &rbacv1.ClusterRole{
		ObjectMeta: metav1.ObjectMeta{
			Name: fmt.Sprintf("%s-%s", nsName, NamespaceEditorRoleSuffix),
		},
		Rules: []rbacv1.PolicyRule{
			{
				Verbs:           []string{"*"},
				APIGroups:       []string{""},
				Resources:       []string{"namespaces"},
				ResourceNames:   []string{nsName},
				NonResourceURLs: []string{},
			},
		},
	}
}

func makeNamespaceEditorRoleBinding(nsName string, groupName string, roleName string) *rbacv1.ClusterRoleBinding {
	return &rbacv1.ClusterRoleBinding{
		ObjectMeta: metav1.ObjectMeta{
			Name: fmt.Sprintf("%s-%s", nsName, NamespaceEditorBindingSuffix),
		},
		RoleRef: rbacv1.RoleRef{
			Kind: "ClusterRole",
			Name: roleName,
		},
		Subjects: []rbacv1.Subject{
			{
				Kind: "Group",
				Name: groupName,
			},
		},
	}
}

// NamespaceReconciler reconciles a Namespace object
type NamespaceReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

func (r *NamespaceReconciler) createIfNotExists(ctx context.Context, log logr.Logger, ns *corev1.Namespace, toFind client.Object, toCreate client.Object) error {
	err := r.Get(ctx, types.NamespacedName{Namespace: toCreate.GetNamespace(), Name: toCreate.GetName()}, toFind)
	if err == nil {
		return nil
	}
	if !errors.IsNotFound(err) {
		return fmt.Errorf("failed to get: %w", err)
	}
	if err = controllerutil.SetControllerReference(ns, toCreate, r.Scheme); err != nil {
		return fmt.Errorf("failed to set controller reference: %w", err)
	}
	err = r.Create(ctx, toCreate)
	if err != nil && !errors.IsAlreadyExists(err) {
		return fmt.Errorf("failed to create: %w", err)
	}
	return nil
}

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Namespace object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.11.0/pkg/reconcile
func (r *NamespaceReconciler) Reconcile(ctx context.Context, req ctrl.Request) (ctrl.Result, error) {
	log := log.FromContext(ctx)
	log.Info("reconcile starting")
	defer log.Info("reconcile finished")

	result := ctrl.Result{
		RequeueAfter: time.Second * 60,
	}

	ns := &corev1.Namespace{}
	if err := r.Get(ctx, req.NamespacedName, ns); err != nil {
		log.Error(err, "unable to fetch namespace")
		err = client.IgnoreNotFound(err)
		if errors.IsNotFound(err) {
			result = ctrl.Result{}
		}
		return result, nil
	}

	abcSlug, exists := ns.Labels[labels.ABC]
	if !exists {
		// TODO(reddi): check special label (for example system: infractl) to determine if ns is InfraCTL-aware.
		// TODO(reddi): check if it's possible to filter objects before reconcile, just on k8s-api querying.
		log.Info("ABC service slug label is not found in labels", "key", labels.ABC)
		return ctrl.Result{}, nil
	}
	abcURL := fmt.Sprintf("svc_%s", strings.ToLower(abcSlug))

	rolesConf := &rolesv1.RolesConfiguration{}
	if err := controllerutil.SetControllerReference(ns, rolesConf, r.Scheme); err != nil {
		log.Error(err, "failed to set controller reference for roles configuration", "name", rolesConf.Name)
		return result, nil
	}
	if err := r.Get(ctx, types.NamespacedName{Namespace: ns.Name, Name: NamespaceRolesConfName}, rolesConf); err != nil {
		if !errors.IsNotFound(err) {
			log.Error(err, "unable to fetch roles configuration", "name", NamespaceRolesConfName)
			return result, nil
		}
		rolesConf.Name = NamespaceRolesConfName
		rolesConf.Namespace = ns.Name
		rolesConf.Spec = &proto_v1.Spec{}
		err := r.Create(ctx, rolesConf)
		if err != nil {
			log.Error(err, "failed to create roles configuration", "name", rolesConf.Name)
			return result, nil
		}
	}

	status := func() *proto_v1.Status {
		binding := makeNamespaceRootRoleBinding(ns.Name, abcURL)
		if err := r.createIfNotExists(ctx, log, ns, &rbacv1.RoleBinding{}, binding); err != nil {
			msg := "failed to create namespace root role binding"
			log.Error(err, msg, "binding", binding.GetName())
			return makeRolesStatusFailure(msg)
		}

		role := makeNamespaceEditorRole(ns.Name)
		if err := r.createIfNotExists(ctx, log, ns, &rbacv1.ClusterRole{}, role); err != nil {
			msg := "failed to create namespace editor role"
			log.Error(err, msg, "role", role.GetName())
			return makeRolesStatusFailure(msg)
		}

		clusterBinding := makeNamespaceEditorRoleBinding(ns.Name, abcURL, role.GetName())
		if err := r.createIfNotExists(ctx, log, ns, &rbacv1.ClusterRoleBinding{}, clusterBinding); err != nil {
			msg := "failed to create namespace editor role binding"
			log.Error(err, msg, "binding", clusterBinding.GetName())
			return makeRolesStatusFailure(msg)
		}
		return makeRolesStatusSuccess("")
	}()

	if !reflect.DeepEqual(rolesConf.Status, status) {
		rolesConf.Status = status
		if err := r.Status().Update(ctx, rolesConf); err != nil {
			log.Error(err, "failed to update roles configuration status", "name", rolesConf.Name)
		}
	}
	return result, nil
}

// SetupWithManager sets up the controller with the Manager.
func (r *NamespaceReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&corev1.Namespace{}).
		Owns(&rbacv1.Role{}).
		Owns(&rbacv1.RoleBinding{}).
		Owns(&rbacv1.ClusterRole{}).
		Owns(&rbacv1.ClusterRoleBinding{}).
		Owns(&rolesv1.RolesConfiguration{}).
		Complete(r)
}
