package v1

import (
	"fmt"
	"regexp"
	"strings"

	"k8s.io/apimachinery/pkg/types"
	"k8s.io/apimachinery/pkg/util/sets"
	"sigs.k8s.io/controller-runtime/pkg/client"

	v1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

var (
	imageRe = regexp.MustCompile(`^([^\.:/]+\.[^/:]+/)?([^:]+):([^@]+)@(.+)+$`)
)

func (o *Runtime) StageSpecNeedsUpdate(_ *v1.DeployStage) bool {
	runtimeGeneration := o.Status.GetSyncStatus().GetStage().GetObservedGeneration()
	return runtimeGeneration < o.Generation
}

func (o *Runtime) CollectSecrets() sets.String {
	result := sets.String{}
	for _, value := range o.GetSpec().GetFiles() {
		if sec, ok := deployutil.ParseSecret(value); ok {
			result.Insert(sec.ID)
		}
	}
	for _, value := range o.GetSpec().GetEnv() {
		if sec, ok := deployutil.ParseSecret(value); ok {
			result.Insert(sec.ID)
		}
	}
	for _, wl := range o.GetSpec().GetWorkloads() {
		for _, value := range wl.GetEnv() {
			if sec, ok := deployutil.ParseSecret(value); ok {
				result.Insert(sec.ID)
			}
		}
	}

	for _, provide := range o.GetSpec().GetProvides() {
		if sec, ok := deployutil.ParseSecret(provide.GetTvm().GetSecret()); ok {
			result.Insert(sec.ID)
		}
	}

	return result

}

func NamespacedName(o client.Object) types.NamespacedName {
	return types.NamespacedName{
		Namespace: o.GetNamespace(),
		Name:      o.GetName(),
	}
}

func SplitDockerImage(image string, err error) (*ypapi.TDockerImageDescription, error) {
	if err != nil {
		return nil, err
	}
	match := imageRe.FindStringSubmatch(image)
	if match == nil {
		return nil, fmt.Errorf("docker image description is broken")
	}
	description := &ypapi.TDockerImageDescription{
		Name:   match[2],
		Tag:    match[3],
		Digest: match[4],
	}
	if len(match[1]) == 0 {
		description.RegistryHost = "registry.yandex.net"
	} else {
		description.RegistryHost = strings.TrimRight(match[1], "/")
	}
	return description, nil
}
