package controllers

import (
	"context"
	"fmt"

	rpv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	psev1 "a.yandex-team.ru/infra/infractl/models/serviceendpoint/proto_v1"
	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type YpPerClusterSettings = ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings
type PerClusterSettings = rpv1.Spec_PerClusterStrategy_ClusterSettings
type MultiCluster = rpv1.Spec_MultiClusterStrategy

const (
	UnknownClusterSettingsEnum int = iota
	BasePerClusterEnum
	ExtendedPerClusterEnum
	MultiClusterEnum
)

func makeImagesForBoxes(kRuntime *rv1.Runtime) (map[string]*ypapi.TDockerImageDescription, error) {
	imagesForBoxes := map[string]*ypapi.TDockerImageDescription{}

	if len(kRuntime.Spec.Image) == 0 {
		return imagesForBoxes, nil
	}

	desc, err := rv1.SplitDockerImage(kRuntime.Spec.Image, nil)
	if err != nil {
		return nil, err
	}
	imagesForBoxes[kRuntime.Name] = desc

	return imagesForBoxes, nil
}

func isEmptyReplicaSet[V any](rs map[string]V) bool {
	return len(rs) == 0
}

func makeDefaultAntiaffinityConstraint() []*ypapi.TAntiaffinityConstraint {
	return []*ypapi.TAntiaffinityConstraint{
		{Key: ptr.String("node"), MaxPods: ptr.Int64(1)},
	}
}

func calculateMaxUnavailable(podCount uint32) uint32 {
	if podCount <= 10 {
		return 1
	}

	return podCount / 10
}

func calculateMaxUnavailableFromSettings(maxAvailSettings *rpv1.Spec_MaxUnavailableSettings, podCount uint32) uint32 {
	var maxUnavailable uint32
	switch s := maxAvailSettings.GetPolicy().(type) {
	case *rpv1.Spec_MaxUnavailableSettings_Count:
		maxUnavailable = s.Count
	case *rpv1.Spec_MaxUnavailableSettings_Percent:
		maxUnavailable = podCount * s.Percent / 100
	}
	return maxUnavailable
}

func newYpPerClusterSettingsMap[V any](m map[string]V) map[string]*YpPerClusterSettings {
	return make(
		map[string]*YpPerClusterSettings,
		len(m),
	)
}

func newYpPerClusterSettings(podCount, maxUnavailable uint32, readyThreshold *ypapi.TDeployReadyCriterion) *YpPerClusterSettings {
	return &YpPerClusterSettings{
		PodCountPolicy: &ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings_PodCount{
			PodCount: podCount,
		},
		DeploymentStrategy: &ypapi.TReplicaSetSpec_TDeploymentStrategy{
			MaxUnavailable: maxUnavailable,
			ReadyCriterion: readyThreshold,
		},
	}
}

func makeDefaultReplicaSetDeploy() *ypapi.TDeployUnitSpec_TReplicaSetDeploy {
	return &ypapi.TDeployUnitSpec_TReplicaSetDeploy{
		ReplicaSetTemplate: &ypapi.TReplicaSetSpec{
			Constraints: &ypapi.TReplicaSetSpec_TConstraints{
				AntiaffinityConstraints: makeDefaultAntiaffinityConstraint(),
			},
		},
	}
}

func fillBasePerClusterReplicaSet(replicas map[string]uint32, rsDeploy *ypapi.TDeployUnitSpec_TReplicaSetDeploy) {
	rsDeploy.PerClusterSettings = newYpPerClusterSettingsMap(replicas)
	for cluster, replicas := range replicas {
		rsDeploy.PerClusterSettings[cluster] = newYpPerClusterSettings(replicas, calculateMaxUnavailable(replicas), nil)
	}
}

func fillExtendedPerClusterReplicaSet(replicas map[string]*PerClusterSettings, rsDeploy *ypapi.TDeployUnitSpec_TReplicaSetDeploy) {
	rsDeploy.PerClusterSettings = newYpPerClusterSettingsMap(replicas)
	for cluster, clusterSettings := range replicas {
		maxUnavailable := calculateMaxUnavailableFromSettings(clusterSettings.MaxUnavailable, clusterSettings.Pods)
		rsDeploy.PerClusterSettings[cluster] = newYpPerClusterSettings(clusterSettings.Pods, maxUnavailable, clusterSettings.ReadyThreshold)
	}
}

func makePerClusterReplicaSet(spec *rpv1.Spec, clusterSettingsType int) *ypapi.TDeployUnitSpec_TReplicaSetDeploy {
	replicaSetDeploy := makeDefaultReplicaSetDeploy()
	switch clusterSettingsType {
	case BasePerClusterEnum:
		fillBasePerClusterReplicaSet(spec.Replicas, replicaSetDeploy)
	case ExtendedPerClusterEnum:
		fillExtendedPerClusterReplicaSet(spec.PerCluster.Replicas, replicaSetDeploy)
	}
	return replicaSetDeploy
}

func makeDefaultMultiClusterReplicaSetDeploy(spec *rpv1.Spec_MultiClusterStrategy) *ypapi.TDeployUnitSpec_TMultiClusterReplicaSetDeploy {
	var totalPods uint32
	for _, replicas := range spec.Replicas {
		totalPods += replicas
	}

	return &ypapi.TDeployUnitSpec_TMultiClusterReplicaSetDeploy{
		ReplicaSet: &ypapi.TMultiClusterReplicaSetSpec{
			DeploymentStrategy: &ypapi.TMultiClusterReplicaSetSpec_TDeploymentStrategy{
				MaxUnavailable: calculateMaxUnavailableFromSettings(spec.MaxUnavailable, totalPods),
			},
		},
	}
}

func makeMultiClusterReplicaSet(spec *rpv1.Spec_MultiClusterStrategy) *ypapi.TDeployUnitSpec_TMultiClusterReplicaSetDeploy {
	multiClusterReplicaSetDeploy := makeDefaultMultiClusterReplicaSetDeploy(spec)

	for cluster, replicas := range spec.Replicas {
		multiClusterReplicaSetDeploy.ReplicaSet.Clusters = append(multiClusterReplicaSetDeploy.ReplicaSet.Clusters, &ypapi.TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences{
			Cluster: cluster,
			Spec: &ypapi.TMultiClusterReplicaSetSpec_TReplicaSetSpecPreferences{
				ReplicaCount: replicas,
				Constraints: &ypapi.TMultiClusterReplicaSetSpec_TConstraints{
					AntiaffinityConstraints: makeDefaultAntiaffinityConstraint(),
				},
			},
		})
	}
	return multiClusterReplicaSetDeploy
}

func processClusterSettings(kRuntime *rv1.Runtime, secrets []*deployutil.Secret, deployUnitSpec *ypapi.TDeployUnitSpec) error {
	pod, podErr := makePod(kRuntime, secrets)
	if podErr != nil {
		return fmt.Errorf("pod spec generation failed: %w", podErr)
	}

	clusterSettingsType := UnknownClusterSettingsEnum
	switch {
	case !isEmptyReplicaSet(kRuntime.Spec.Replicas):
		clusterSettingsType = BasePerClusterEnum
	case kRuntime.Spec.PerCluster != nil && !isEmptyReplicaSet(kRuntime.Spec.PerCluster.Replicas):
		clusterSettingsType = ExtendedPerClusterEnum
	case kRuntime.Spec.MultiCluster != nil && !isEmptyReplicaSet(kRuntime.Spec.MultiCluster.Replicas):
		clusterSettingsType = MultiClusterEnum
	}
	if clusterSettingsType == UnknownClusterSettingsEnum {
		return fmt.Errorf("no cluster settings specified")
	}

	if clusterSettingsType == BasePerClusterEnum || clusterSettingsType == ExtendedPerClusterEnum {
		replicaSet := makePerClusterReplicaSet(kRuntime.Spec, clusterSettingsType)
		replicaSet.ReplicaSetTemplate.PodTemplateSpec = &ypapi.TPodTemplateSpec{
			Spec: pod,
		}
		deployUnitSpec.PodDeployPrimitive = &ypapi.TDeployUnitSpec_ReplicaSet{
			ReplicaSet: replicaSet,
		}
	} else if clusterSettingsType == MultiClusterEnum {
		multiClusterReplicaSet := makeMultiClusterReplicaSet(kRuntime.Spec.MultiCluster)
		multiClusterReplicaSet.ReplicaSet.PodTemplateSpec = &ypapi.TPodTemplateSpec{
			Spec: pod,
		}
		deployUnitSpec.PodDeployPrimitive = &ypapi.TDeployUnitSpec_MultiClusterReplicaSet{
			MultiClusterReplicaSet: multiClusterReplicaSet,
		}
	}
	return nil
}

type makeDeployUnitResult struct {
	DeployUnit *ypapi.TDeployUnitSpec
	Provides   *psev1.Spec
}

func (b *stageBuilder) makeDeployUnit(ctx context.Context, kRuntime *rv1.Runtime) (makeDeployUnitResult, error) {
	var result makeDeployUnitResult

	if len(kRuntime.Spec.Image) != 0 && len(kRuntime.Spec.Layers) != 0 {
		return result, fmt.Errorf("both image and layers are not empty")
	}

	imagesForBoxes, err := makeImagesForBoxes(kRuntime)
	if err != nil {
		return result, fmt.Errorf("filesystem image generation failed: %w", err)
	}

	tvmConfig, err := b.makeTvmConfig(ctx, kRuntime)
	if err != nil {
		return result, fmt.Errorf("tvm config generation failed: %w", err)
	}

	result.DeployUnit = &ypapi.TDeployUnitSpec{
		CollectPortometricsFromSidecars: true,
		DeploySettings: &ypapi.TDeployUnitSpec_TDeploySettings{
			DeployStrategy: ypapi.TDeployUnitSpec_TDeploySettings_SEQUENTIAL,
		},
		ImagesForBoxes: imagesForBoxes,
		LogbrokerConfig: &ypapi.TLogbrokerConfig{
			PodAdditionalResourcesRequest: &ypapi.TPodSpec_TResourceRequests{
				VcpuGuarantee: ptr.Uint64(0),
				VcpuLimit:     ptr.Uint64(0),
			},
		},
		NetworkDefaults: &ypapi.TNetworkDefaults{
			// Even though separate object for network macro exists, client must fill it anyway
			NetworkId: kRuntime.Spec.NetworkId,
		},
		TvmConfig: tvmConfig.Config,
	}
	result.Provides = tvmConfig.Provides

	err = processClusterSettings(kRuntime, tvmConfig.Secrets, result.DeployUnit)
	if err != nil {
		return result, fmt.Errorf("cluster settings processing failed: %w", err)
	}
	return result, nil
}
