package controllers

import (
	"context"
	"sort"
	"testing"

	"github.com/stretchr/testify/require"
	v1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	v1proto "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	deployinfrav1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

func TestMakeImagesForBoxes(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		testCases := []string{
			"invalid",
			"test/image:1.0",
			"test/image:1.0@",
		}

		for _, tc := range testCases {
			t.Run(tc, func(t *testing.T) {
				_, err := makeImagesForBoxes(&deployinfrav1.Runtime{
					Spec: &v1proto.Spec{
						Image: tc,
					},
				})
				require.Error(t, err)
			})
		}
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		images, err := makeImagesForBoxes(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Image: "test/image:1.0@sha256:aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
			},
			ObjectMeta: v1.ObjectMeta{Name: "test"},
		})
		require.NoError(t, err)

		requireProtoEqual(
			t,
			map[string]*ypapi.TDockerImageDescription{
				"test": {
					Name:         "test/image",
					RegistryHost: "registry.yandex.net",
					Tag:          "1.0",
					Digest:       "sha256:aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
				},
			},
			images,
		)
	})

	t.Run("empty image", func(t *testing.T) {
		t.Parallel()

		images, err := makeImagesForBoxes(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{},
		})
		require.NoError(t, err)
		require.Equal(t, 0, len(images))
	})
}

const DefaultMaxUnavailable = 1

func TestMakeReplicaSet(t *testing.T) {
	t.Parallel()

	t.Run("base per cluster ok", func(t *testing.T) {
		t.Parallel()

		const sasPodCount, vlaPodCount = 2, 3
		rs := makePerClusterReplicaSet(&v1proto.Spec{
			Replicas: map[string]uint32{"sas": sasPodCount, "vla": vlaPodCount},
			Command:  "cmd",
		}, BasePerClusterEnum)

		requireProtoEqual(
			t,
			map[string]*ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings{
				"sas": {
					PodCountPolicy: &ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings_PodCount{
						PodCount: sasPodCount,
					},
					DeploymentStrategy: &ypapi.TReplicaSetSpec_TDeploymentStrategy{
						MaxUnavailable: DefaultMaxUnavailable,
					},
				},
				"vla": {
					PodCountPolicy: &ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings_PodCount{
						PodCount: vlaPodCount,
					},
					DeploymentStrategy: &ypapi.TReplicaSetSpec_TDeploymentStrategy{
						MaxUnavailable: DefaultMaxUnavailable,
					},
				},
			},
			rs.PerClusterSettings,
		)
	})

	t.Run("extended per cluster ok", func(t *testing.T) {
		t.Parallel()

		const sasPodCount, vlaPodCount = 3, 4
		const countMaxUnavailable, percentMaxUnavailable = 2, 50
		rs := makePerClusterReplicaSet(&v1proto.Spec{
			PerCluster: &v1proto.Spec_PerClusterStrategy{
				Replicas: map[string]*v1proto.Spec_PerClusterStrategy_ClusterSettings{
					"sas": &v1proto.Spec_PerClusterStrategy_ClusterSettings{
						Pods: sasPodCount,
						MaxUnavailable: &v1proto.Spec_MaxUnavailableSettings{
							Policy: &v1proto.Spec_MaxUnavailableSettings_Count{
								Count: countMaxUnavailable,
							},
						},
					},
					"vla": &v1proto.Spec_PerClusterStrategy_ClusterSettings{
						Pods: vlaPodCount,
						MaxUnavailable: &v1proto.Spec_MaxUnavailableSettings{
							Policy: &v1proto.Spec_MaxUnavailableSettings_Percent{
								Percent: percentMaxUnavailable,
							},
						},
					},
				},
			},
		}, ExtendedPerClusterEnum)

		requireProtoEqual(
			t,
			map[string]*ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings{
				"sas": {
					PodCountPolicy: &ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings_PodCount{
						PodCount: sasPodCount,
					},
					DeploymentStrategy: &ypapi.TReplicaSetSpec_TDeploymentStrategy{
						MaxUnavailable: countMaxUnavailable,
					},
				},
				"vla": {
					PodCountPolicy: &ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings_PodCount{
						PodCount: vlaPodCount,
					},
					DeploymentStrategy: &ypapi.TReplicaSetSpec_TDeploymentStrategy{
						MaxUnavailable: vlaPodCount * percentMaxUnavailable / 100,
					},
				},
			},
			rs.PerClusterSettings,
		)
	})

	t.Run("multi cluster, count max unavailable ok", func(t *testing.T) {
		t.Parallel()

		const sasPodCount, vlaPodCount = 3, 4
		const countMaxUnavailable = 3
		rs := makeMultiClusterReplicaSet(&v1proto.Spec_MultiClusterStrategy{
			Replicas: map[string]uint32{"sas": sasPodCount, "vla": vlaPodCount},
			MaxUnavailable: &v1proto.Spec_MaxUnavailableSettings{
				Policy: &v1proto.Spec_MaxUnavailableSettings_Count{
					Count: countMaxUnavailable,
				},
			},
		})

		sort.Slice(rs.ReplicaSet.Clusters, func(i, j int) bool {
			return rs.ReplicaSet.Clusters[i].Cluster < rs.ReplicaSet.Clusters[j].Cluster
		})
		requireProtoEqual(
			t,
			&ypapi.TMultiClusterReplicaSetSpec{
				Clusters: []*ypapi.TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences{
					{
						Cluster: "sas", Spec: &ypapi.TMultiClusterReplicaSetSpec_TReplicaSetSpecPreferences{
							ReplicaCount: sasPodCount,
							Constraints: &ypapi.TMultiClusterReplicaSetSpec_TConstraints{
								AntiaffinityConstraints: makeDefaultAntiaffinityConstraint()},
						}},
					{
						Cluster: "vla", Spec: &ypapi.TMultiClusterReplicaSetSpec_TReplicaSetSpecPreferences{
							ReplicaCount: vlaPodCount,
							Constraints: &ypapi.TMultiClusterReplicaSetSpec_TConstraints{
								AntiaffinityConstraints: makeDefaultAntiaffinityConstraint()},
						}},
				},
				DeploymentStrategy: &ypapi.TMultiClusterReplicaSetSpec_TDeploymentStrategy{
					MaxUnavailable: countMaxUnavailable,
				},
			},
			rs.ReplicaSet,
		)
	})

	t.Run("multi cluster, percent max unavailable ok", func(t *testing.T) {
		t.Parallel()

		const sasPodCount, vlaPodCount = 2, 6
		const percentMaxUnavailable = 50
		rs := makeMultiClusterReplicaSet(&v1proto.Spec_MultiClusterStrategy{
			Replicas: map[string]uint32{"sas": sasPodCount, "vla": vlaPodCount},
			MaxUnavailable: &v1proto.Spec_MaxUnavailableSettings{
				Policy: &v1proto.Spec_MaxUnavailableSettings_Percent{
					Percent: percentMaxUnavailable,
				},
			},
		})

		sort.Slice(rs.ReplicaSet.Clusters, func(i, j int) bool {
			return rs.ReplicaSet.Clusters[i].Cluster < rs.ReplicaSet.Clusters[j].Cluster
		})
		requireProtoEqual(
			t,
			&ypapi.TMultiClusterReplicaSetSpec{
				Clusters: []*ypapi.TMultiClusterReplicaSetSpec_TClusterReplicaSetSpecPreferences{
					{
						Cluster: "sas", Spec: &ypapi.TMultiClusterReplicaSetSpec_TReplicaSetSpecPreferences{
							ReplicaCount: sasPodCount,
							Constraints: &ypapi.TMultiClusterReplicaSetSpec_TConstraints{
								AntiaffinityConstraints: makeDefaultAntiaffinityConstraint()},
						}},
					{
						Cluster: "vla", Spec: &ypapi.TMultiClusterReplicaSetSpec_TReplicaSetSpecPreferences{
							ReplicaCount: vlaPodCount,
							Constraints: &ypapi.TMultiClusterReplicaSetSpec_TConstraints{
								AntiaffinityConstraints: makeDefaultAntiaffinityConstraint()},
						}},
				},
				DeploymentStrategy: &ypapi.TMultiClusterReplicaSetSpec_TDeploymentStrategy{
					MaxUnavailable: (sasPodCount + vlaPodCount) * percentMaxUnavailable / 100,
				},
			},
			rs.ReplicaSet,
		)
	})
}

func TestMakeDeployUnit(t *testing.T) {
	t.Parallel()

	builder := stageBuilder{nil}

	t.Run("minimal ok", func(t *testing.T) {
		t.Parallel()

		_, err := builder.makeDeployUnit(context.TODO(), &deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Replicas: map[string]uint32{"sas": 1},
				Compute:  &v1proto.Spec_ComputeResources{},
			},
		})
		require.NoError(t, err)
	})

	t.Run("incorrect pod", func(t *testing.T) {
		t.Parallel()

		_, err := builder.makeDeployUnit(context.TODO(), &deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Replicas: map[string]uint32{"sas": 1},
				Compute:  &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
			},
		})
		require.Error(t, err)
	})

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name string
			spec *v1proto.Spec
		}{
			{
				name: "invalid image",
				spec: &v1proto.Spec{
					Replicas: map[string]uint32{"sas": 1},
					Image:    "invalid",
				},
			},
			{
				name: "invalid rs",
				spec: &v1proto.Spec{
					Replicas: map[string]uint32{"sas": 1},
					Image:    "test/image:1.0@sha256:aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
					Compute:  &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
				},
			},
			{
				name: "image and layers",
				spec: &v1proto.Spec{
					Replicas: map[string]uint32{"sas": 1},
					Image:    "test/image:1.0@sha256:aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
					Compute:  &v1proto.Spec_ComputeResources{},
					Layers: []*v1proto.Spec_PortoLayer{
						{
							Id:         "layer",
							ResourceId: "sbr:123",
						},
					},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()

				_, err := builder.makeDeployUnit(context.TODO(), &deployinfrav1.Runtime{
					Spec: tc.spec,
				})
				require.Error(t, err)
			})
		}
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		duResult, err := builder.makeDeployUnit(context.TODO(), &deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Replicas: map[string]uint32{"sas": 1},
				Image:    "test/image:1.0@sha256:aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
				Compute:  &v1proto.Spec_ComputeResources{},
				//Deploy: &v1proto.Spec_DeployStrategy{
				//	MaxUnavailable: 1,
				//},
				NetworkId: "_TEST_",
			},
			ObjectMeta: v1.ObjectMeta{Name: "test"},
		})
		require.NoError(t, err)

		requireProtoEqual(
			t,
			map[string]*ypapi.TDockerImageDescription{
				"test": {
					Name:         "test/image",
					RegistryHost: "registry.yandex.net",
					Tag:          "1.0",
					Digest:       "sha256:aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
				},
			},
			duResult.DeployUnit.ImagesForBoxes,
		)

		requireProtoEqual(
			t,
			&ypapi.TDeployUnitSpec_TDeploySettings{
				DeployStrategy: ypapi.TDeployUnitSpec_TDeploySettings_SEQUENTIAL,
			},
			duResult.DeployUnit.DeploySettings,
		)

		requireProtoEqual(
			t,
			&ypapi.TNetworkDefaults{
				NetworkId: "_TEST_",
			},
			duResult.DeployUnit.NetworkDefaults,
		)
	})
}
