package controllers

import (
	"fmt"
	"sort"

	prv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/infra/infractl/internal/quantities"
	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/proto/core/ytree"
	"a.yandex-team.ru/yt/go/yson"
)

type Attribute struct {
	Key   string
	Value any
}

func makeAttributesMap(attrs []*Attribute) (*ytree.TAttributeDictionary, error) {
	if attrs == nil {
		return nil, nil
	}

	ret := &ytree.TAttributeDictionary{
		Attributes: make([]*ytree.TAttribute, 0, len(attrs)),
	}

	for _, attr := range attrs {
		valueBytes, err := yson.Marshal(attr.Value)
		if err != nil {
			return nil, err
		}

		ret.Attributes = append(ret.Attributes, &ytree.TAttribute{
			Key:   ptr.String(attr.Key),
			Value: valueBytes,
		})
	}

	return ret, nil
}

func makeDiskVolumeRequests(storage *prv1.Spec_Storage) ([]*ypapi.TPodSpec_TDiskVolumeRequest, error) {
	attributes, err := makeAttributesMap([]*Attribute{
		{
			Key:   "used_by_infra",
			Value: ptr.Bool(true),
		},
	})
	if err != nil {
		return nil, err
	}

	storageClass := storage.GetStorageClass()
	if len(storageClass) == 0 {
		storageClass = "hdd"
	}

	quota, err := quantities.ConvertFromHuman(storage.GetQuota())
	if err != nil {
		return nil, fmt.Errorf("invalid quota: %w", err)
	}
	bandwidthGuarantee, err := quantities.ConvertFromHuman(storage.GetIoBandwidth())
	if err != nil {
		return nil, fmt.Errorf("invalid io bandwidth: %w", err)
	}

	var bandwidthLimit uint64
	switch storageClass {
	case "hdd":
		bandwidthLimit = bandwidthGuarantee * 2
	default:
		bandwidthLimit = bandwidthGuarantee
	}

	return []*ypapi.TPodSpec_TDiskVolumeRequest{{
		Id:           ptr.String("disk-0"),
		Labels:       attributes,
		StorageClass: &storageClass,
		ConcretePolicy: &ypapi.TPodSpec_TDiskVolumeRequest_QuotaPolicy{
			QuotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
				Capacity:           ptr.Uint64(quota),
				BandwidthGuarantee: ptr.Uint64(bandwidthGuarantee),
				BandwidthLimit:     ptr.Uint64(bandwidthLimit),
			},
		},
	}}, nil
}

func makeResourceRequests(spec *prv1.Spec) (*ypapi.TPodSpec_TResourceRequests, error) {
	rr := &ypapi.TPodSpec_TResourceRequests{}

	res := spec.GetCompute()
	vcpu, err := quantities.ConvertFromHuman(res.GetVcpu())
	if err != nil {
		return nil, fmt.Errorf("invalid vcpu: %w", err)
	}
	memory, err := quantities.ConvertFromHuman(res.GetMemory())
	if err != nil {
		return nil, fmt.Errorf("invalid memory: %w", err)
	}
	net, err := quantities.ConvertFromHuman(res.GetNet())
	if err != nil {
		return nil, fmt.Errorf("invalid net: %w", err)
	}

	rr.VcpuLimit = ptr.Uint64(vcpu)
	rr.VcpuGuarantee = rr.VcpuLimit
	rr.MemoryLimit = ptr.Uint64(memory)
	rr.MemoryGuarantee = rr.MemoryLimit
	rr.NetworkBandwidthGuarantee = ptr.Uint64(net)

	return rr, nil
}

func makeVolumes(
	specVolumes map[string]*prv1.Spec_Volume,
) ([]*podagent.TVolume, []*podagent.TMountedVolume) {
	volumes := make([]*podagent.TVolume, 0, len(specVolumes))
	mounts := make([]*podagent.TMountedVolume, 0, len(specVolumes))

	for mountPath, volume := range specVolumes {
		id := makeIDFromMountPath(mountPath)
		volumes = append(volumes, &podagent.TVolume{
			Id:              id,
			PersistenceType: podagent.EVolumePersistenceType_EVolumePersistenceType_PERSISTENT,
		})

		var mode podagent.EVolumeMountMode
		switch volume.Mode {
		case "rw":
			mode = podagent.EVolumeMountMode_EVolumeMountMode_READ_WRITE
		case "ro":
			mode = podagent.EVolumeMountMode_EVolumeMountMode_READ_ONLY
		default:
			mode = podagent.EVolumeMountMode_EVolumeMountMode_READ_WRITE
		}
		mounts = append(mounts, &podagent.TMountedVolume{
			VolumeType: &podagent.TMountedVolume_VolumeRef{VolumeRef: id},
			MountPoint: mountPath,
			Mode:       mode,
		})
	}

	sort.Slice(volumes, func(i, j int) bool {
		return volumes[i].Id < volumes[j].Id
	})
	sort.Slice(mounts, func(i, j int) bool {
		return mounts[i].GetVolumeRef() < mounts[j].GetVolumeRef()
	})

	return volumes, mounts
}

func makePod(kRuntime *rv1.Runtime, secrets []*deployutil.Secret) (*ypapi.TPodSpec, error) {
	resources := makeResources(
		kRuntime.Spec.Files,
		kRuntime.Spec.Resources,
		kRuntime.Spec.Layers,
	)

	diskVolumeRequests, err := makeDiskVolumeRequests(kRuntime.Spec.Storage)
	if err != nil {
		return nil, fmt.Errorf("disk volume request generation failed: %w", err)
	}

	volumes, mountedVolumes := makeVolumes(kRuntime.Spec.Volumes)

	resourceRequests, err := makeResourceRequests(kRuntime.Spec)
	if err != nil {
		return nil, fmt.Errorf("resource requests generation failed: %w", err)
	}

	envVars, secretRefs := deployutil.ParseEnvValues(kRuntime.Spec.Env, true)

	workloads, workloadsSecretRefs, err := makeWorkloads(kRuntime)
	if err != nil {
		return nil, fmt.Errorf("workloads generation failed: %w", err)
	}

	for alias, ref := range workloadsSecretRefs {
		secretRefs[alias] = ref
	}
	for alias, ref := range resources.secretRefs {
		secretRefs[alias] = ref
	}

	for _, secret := range secrets {
		secretRefs[secret.MakeAlias()] = secret.MakeSecretRef()
	}

	mutableWorkloads := make([]*podagent.TMutableWorkload, 0, len(workloads))
	for _, workload := range workloads {
		mutableWorkloads = append(mutableWorkloads, &podagent.TMutableWorkload{
			WorkloadRef: workload.Id,
		})
	}

	layerRefs := make([]string, 0, len(resources.resources.Layers))
	for _, layer := range resources.resources.Layers {
		layerRefs = append(layerRefs, layer.Id)
	}

	return &ypapi.TPodSpec{
		PodAgentPayload: &ypapi.TPodSpec_TPodAgentPayload{
			Spec: &podagent.TPodAgentSpec{
				MutableWorkloads: mutableWorkloads,
				Workloads:        workloads,
				Boxes: []*podagent.TBox{
					{
						Id:                kRuntime.Name,
						IsolationMode:     podagent.EContainerIsolationMode_EContainerIsolationMode_CHILD_ONLY,
						CgroupFsMountMode: podagent.ECgroupFsMountMode_ECgroupFsMountMode_RO,
						StaticResources:   resources.mounts,
						Env:               envVars,
						Volumes:           mountedVolumes,
						Rootfs:            &podagent.TRootfsVolume{LayerRefs: layerRefs},
					},
				},
				Resources: resources.resources,
				Volumes:   volumes,
			},
		},
		ResourceRequests:   resourceRequests,
		SecretRefs:         secretRefs,
		DiskVolumeRequests: diskVolumeRequests,
	}, nil
}
