package controllers

import (
	"testing"

	"github.com/stretchr/testify/require"

	v1proto "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	deployinfrav1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"a.yandex-team.ru/yt/go/proto/core/ytree"
)

func TestMakeAttributesMap(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()
		_, err := makeAttributesMap([]*Attribute{
			{"key", func() {}},
		})
		require.Error(t, err)
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()
		attrMap, err := makeAttributesMap([]*Attribute{
			{"key1", true},
			{"key2", 42},
			{"key3", struct {
				A string
				B string
			}{A: "test", B: "struct"}},
		})
		require.NoError(t, err)
		requireProtoEqual(
			t,
			&ytree.TAttributeDictionary{
				Attributes: []*ytree.TAttribute{
					{Key: ptr.String("key1"), Value: []byte("%true")},
					{Key: ptr.String("key2"), Value: []byte("42")},
					{Key: ptr.String("key3"), Value: []byte("{A=test;B=struct;}")},
				},
			},
			attrMap,
		)
	})
}

func TestMakeDiskVolumeRequests(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()
		testCases := []struct {
			name string
			spec *v1proto.Spec_Storage
		}{
			{
				name: "invalid quota",
				spec: &v1proto.Spec_Storage{Quota: "invalid"},
			},
			{
				name: "invalid bandwidth",
				spec: &v1proto.Spec_Storage{IoBandwidth: "invalid"},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				_, err := makeDiskVolumeRequests(tc.spec)
				require.Error(t, err)
			})
		}
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()
		testCases := []struct {
			name         string
			spec         *v1proto.Spec_Storage
			storageClass *string
			quotaPolicy  *ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy
		}{
			{
				name:         "nil storage",
				spec:         nil,
				storageClass: ptr.String("hdd"),
				quotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
					Capacity:           ptr.Uint64(0),
					BandwidthGuarantee: ptr.Uint64(0),
					BandwidthLimit:     ptr.Uint64(0),
				},
			},
			{
				name: "explicit storage class",
				spec: &v1proto.Spec_Storage{
					StorageClass: "ssd",
					IoBandwidth:  "2",
				},
				storageClass: ptr.String("ssd"),
				quotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
					Capacity:           ptr.Uint64(0),
					BandwidthGuarantee: ptr.Uint64(2),
					BandwidthLimit:     ptr.Uint64(2),
				},
			},
			{
				name: "implicit storage class",
				spec: &v1proto.Spec_Storage{
					Quota:       "2",
					IoBandwidth: "3",
				},
				storageClass: ptr.String("hdd"),
				quotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
					Capacity:           ptr.Uint64(2),
					BandwidthGuarantee: ptr.Uint64(3),
					BandwidthLimit:     ptr.Uint64(6),
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				requests, err := makeDiskVolumeRequests(tc.spec)
				require.NoError(t, err)
				requireProtoEqual(
					t,
					&ypapi.TPodSpec_TDiskVolumeRequest{
						Id: ptr.String("disk-0"),
						Labels: &ytree.TAttributeDictionary{
							Attributes: []*ytree.TAttribute{
								{Key: ptr.String("used_by_infra"), Value: []byte("%true")},
							},
						},
						StorageClass: tc.storageClass,
						ConcretePolicy: &ypapi.TPodSpec_TDiskVolumeRequest_QuotaPolicy{
							QuotaPolicy: tc.quotaPolicy,
						},
					},
					requests[0],
				)
			})
		}
	})
}

func TestMakeResourceRequests(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name    string
			spec    *v1proto.Spec
			request *ypapi.TPodSpec_TResourceRequests
		}{
			{
				name: "invalid vcpu",
				spec: &v1proto.Spec{
					Compute: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
				},
			},
			{
				name: "invalid memory",
				spec: &v1proto.Spec{
					Compute: &v1proto.Spec_ComputeResources{Memory: "invalid"},
				},
			},
			{
				name: "invalid net",
				spec: &v1proto.Spec{
					Compute: &v1proto.Spec_ComputeResources{Net: "invalid"},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				_, err := makeResourceRequests(tc.spec)
				require.Error(t, err)
			})
		}
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()
		request, err := makeResourceRequests(&v1proto.Spec{
			Compute: &v1proto.Spec_ComputeResources{
				Vcpu:   "1",
				Memory: "2",
				Net:    "3",
			},
		})
		require.NoError(t, err)
		requireProtoEqual(
			t,
			&ypapi.TPodSpec_TResourceRequests{
				VcpuLimit:                 ptr.Uint64(1),
				VcpuGuarantee:             ptr.Uint64(1),
				MemoryLimit:               ptr.Uint64(2),
				MemoryGuarantee:           ptr.Uint64(2),
				NetworkBandwidthGuarantee: ptr.Uint64(3),
			},
			request,
		)
	})
}

func TestMakeVolumes(t *testing.T) {
	t.Parallel()

	podagentVolumes, mountedVolumes := makeVolumes(map[string]*v1proto.Spec_Volume{
		"/test1": {Mode: "rw"},
		"/test2": {Mode: "ro"},
		"/test3": {Mode: ""},
	})

	requireProtoEqual(
		t,
		[]*podagent.TVolume{
			{
				Id:              "test1",
				PersistenceType: podagent.EVolumePersistenceType_EVolumePersistenceType_PERSISTENT,
			},
			{
				Id:              "test2",
				PersistenceType: podagent.EVolumePersistenceType_EVolumePersistenceType_PERSISTENT,
			},
			{
				Id:              "test3",
				PersistenceType: podagent.EVolumePersistenceType_EVolumePersistenceType_PERSISTENT,
			},
		},
		podagentVolumes,
	)
	requireProtoEqual(
		t,
		[]*podagent.TMountedVolume{
			{
				VolumeType: &podagent.TMountedVolume_VolumeRef{VolumeRef: "test1"},
				MountPoint: "/test1",
				Mode:       podagent.EVolumeMountMode_EVolumeMountMode_READ_WRITE,
			},
			{
				VolumeType: &podagent.TMountedVolume_VolumeRef{VolumeRef: "test2"},
				MountPoint: "/test2",
				Mode:       podagent.EVolumeMountMode_EVolumeMountMode_READ_ONLY,
			},
			{
				VolumeType: &podagent.TMountedVolume_VolumeRef{VolumeRef: "test3"},
				MountPoint: "/test3",
				Mode:       podagent.EVolumeMountMode_EVolumeMountMode_READ_WRITE,
			},
		},
		mountedVolumes,
	)
}

func TestMakePod(t *testing.T) {
	t.Parallel()

	// to ensure invalid... tests break in correct places
	t.Run("minimal ok", func(t *testing.T) {
		_, err := makePod(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{Compute: &v1proto.Spec_ComputeResources{}},
		}, nil)
		require.NoError(t, err)
	})

	t.Run("with errors", func(t *testing.T) {
		t.Parallel()
		testCases := []struct {
			name string
			spec *v1proto.Spec
			pod  *ypapi.TPodSpec
		}{
			{
				name: "invalid disk volume requests",
				spec: &v1proto.Spec{
					Storage: &v1proto.Spec_Storage{Quota: "invalid"},
					Compute: &v1proto.Spec_ComputeResources{},
				},
			},
			{
				name: "invalid resource requests",
				spec: &v1proto.Spec{
					Compute: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
				},
			},
			{
				name: "invalid workloads",
				spec: &v1proto.Spec{
					Compute: &v1proto.Spec_ComputeResources{},
					Workloads: map[string]*v1proto.Spec_Workload{
						"workload": {},
					},
					Command: "cmd",
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				_, err := makePod(&deployinfrav1.Runtime{Spec: tc.spec}, nil)
				require.Error(t, err)
			})
		}
	})

	t.Run("resources", func(t *testing.T) {
		t.Parallel()
		pod, err := makePod(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Compute: &v1proto.Spec_ComputeResources{},
				Files: map[string]string{
					"/test/a": "b",
				},
			},
		}, nil)
		require.NoError(t, err)

		requireProtoEqual(t, &podagent.TResourceGang{
			StaticResources: []*podagent.TResource{
				{
					Id: "test",
					DownloadMethod: &podagent.TResource_Files{
						Files: &podagent.TFiles{
							Files: []*podagent.TFile{
								{
									FileName: "a",
									Content: &podagent.TFile_RawData{
										RawData: "b",
									},
								},
							},
						},
					},
					Verification: &podagent.TVerification{Disabled: true},
				},
			},
		}, pod.PodAgentPayload.Spec.Resources)
		requireProtoEqual(
			t,
			[]*podagent.TMountedStaticResource{
				{
					ResourceRef: "test",
					MountPoint:  "/test/",
				},
			},
			pod.PodAgentPayload.Spec.Boxes[0].StaticResources,
		)
	})

	t.Run("disk volume", func(t *testing.T) {
		t.Parallel()
		pod, err := makePod(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Compute: &v1proto.Spec_ComputeResources{},
				Storage: &v1proto.Spec_Storage{Quota: "1"},
			},
		}, nil)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			[]*ypapi.TPodSpec_TDiskVolumeRequest{
				{
					Id: ptr.String("disk-0"),
					Labels: &ytree.TAttributeDictionary{
						Attributes: []*ytree.TAttribute{
							{Key: ptr.String("used_by_infra"), Value: []byte("%true")},
						},
					},
					StorageClass: ptr.String("hdd"),
					ConcretePolicy: &ypapi.TPodSpec_TDiskVolumeRequest_QuotaPolicy{
						QuotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
							Capacity:           ptr.Uint64(1),
							BandwidthGuarantee: ptr.Uint64(0),
							BandwidthLimit:     ptr.Uint64(0),
						},
					},
				},
			},
			pod.DiskVolumeRequests,
		)
	})

	t.Run("volumes", func(t *testing.T) {
		t.Parallel()
		pod, err := makePod(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Compute: &v1proto.Spec_ComputeResources{},
				Volumes: map[string]*v1proto.Spec_Volume{
					"/test": {Mode: "rw"},
				},
			},
		}, nil)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			[]*podagent.TMountedVolume{
				{
					VolumeType: &podagent.TMountedVolume_VolumeRef{VolumeRef: "test"},
					MountPoint: "/test",
					Mode:       podagent.EVolumeMountMode_EVolumeMountMode_READ_WRITE,
				},
			},
			pod.PodAgentPayload.Spec.Boxes[0].Volumes,
		)
		requireProtoEqual(
			t,
			[]*podagent.TVolume{
				{
					Id:              "test",
					PersistenceType: podagent.EVolumePersistenceType_EVolumePersistenceType_PERSISTENT,
				},
			},
			pod.PodAgentPayload.Spec.Volumes,
		)
	})

	t.Run("resource requests", func(t *testing.T) {
		t.Parallel()
		pod, err := makePod(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Compute: &v1proto.Spec_ComputeResources{Vcpu: "1"},
			},
		}, nil)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			&ypapi.TPodSpec_TResourceRequests{
				VcpuLimit:                 ptr.Uint64(1),
				VcpuGuarantee:             ptr.Uint64(1),
				MemoryLimit:               ptr.Uint64(0),
				MemoryGuarantee:           ptr.Uint64(0),
				NetworkBandwidthGuarantee: ptr.Uint64(0),
			},
			pod.ResourceRequests,
		)
	})

	t.Run("env vars", func(t *testing.T) {
		t.Parallel()
		pod, err := makePod(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Compute: &v1proto.Spec_ComputeResources{},
				Env:     map[string]string{"a": "b"},
			},
		}, nil)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			[]*podagent.TEnvVar{
				{
					Name: "a",
					Value: &podagent.TEnvVarValue{
						Value: &podagent.TEnvVarValue_LiteralEnv{
							LiteralEnv: &podagent.LiteralEnvSelector{Value: "b"},
						},
					},
				},
			},
			pod.PodAgentPayload.Spec.Boxes[0].Env,
		)
	})

	t.Run("workloads", func(t *testing.T) {
		t.Parallel()
		pod, err := makePod(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Compute: &v1proto.Spec_ComputeResources{},
				Workloads: map[string]*v1proto.Spec_Workload{
					"workload": {
						Command: &v1proto.Spec_Command{Cmd: "cmd"},
					},
				},
			},
		}, nil)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			[]*podagent.TWorkload{
				{
					Id: "workload",
					Start: &podagent.TUtilityContainer{
						ComputeResources: &podagent.TComputeResources{},
						CommandLine:      "cmd",
					},
					ReadinessCheck: DefaultReadinessCheck,
					LivenessCheck:  DefaultLivenessCheck,
					TransmitLogs:   true,
				},
			},
			pod.PodAgentPayload.Spec.Workloads,
		)
		requireProtoEqual(
			t,
			[]*podagent.TMutableWorkload{
				{
					WorkloadRef: "workload",
				},
			},
			pod.PodAgentPayload.Spec.MutableWorkloads,
		)
	})

	t.Run("merge secret refs", func(t *testing.T) {
		t.Parallel()
		pod, err := makePod(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Compute: &v1proto.Spec_ComputeResources{},
				Env: map[string]string{
					"secret1": "${s1:v1:k1}",
				},
				Workloads: map[string]*v1proto.Spec_Workload{
					"workload": {
						Env: map[string]string{
							"secret2": "${s2:v2:k2}",
						},
					},
				},
				Files: map[string]string{
					"/test/secret3": "${s3:v3:k3}",
				},
			},
		}, nil)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			map[string]*ypapi.TSecretRef{
				"s1:v1": {
					SecretId:      ptr.String("s1"),
					SecretVersion: ptr.String("v1"),
				},
				"s2:v2": {
					SecretId:      ptr.String("s2"),
					SecretVersion: ptr.String("v2"),
				},
				"s3:v3": {
					SecretId:      ptr.String("s3"),
					SecretVersion: ptr.String("v3"),
				},
			},
			pod.SecretRefs,
		)
	})
}
