package controllers

import (
	"fmt"
	"path"
	"sort"
	"strings"

	v1proto "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

func makeIDFromMountPath(mountPath string) string {
	if mountPath == "/" || len(mountPath) == 0 {
		return "-"
	}
	return strings.ReplaceAll(strings.Trim(mountPath, "/"), "/", "-")
}

func makeFilesStaticResources(
	files map[string]string,
) (map[string]*podagent.TResource, map[string]*ypapi.TSecretRef) {
	secretRefs := map[string]*ypapi.TSecretRef{}
	resourcesMap := map[string]*podagent.TResource{}

	for filePath, content := range files {
		mountPath, name := path.Split(filePath)

		podagentFile := &podagent.TFile{FileName: name}
		secret, ok := deployutil.ParseSecret(content)
		if ok {
			alias := secret.MakeAlias()
			podagentFile.Content = &podagent.TFile_SecretData{
				SecretData: &podagent.SecretSelector{
					Alias: alias,
					Id:    secret.Key,
				},
			}
			secretRefs[alias] = &ypapi.TSecretRef{
				SecretId:      &secret.ID,
				SecretVersion: &secret.Version,
			}
		} else {
			podagentFile.Content = &podagent.TFile_RawData{
				RawData: string(content),
			}
		}

		resource, ok := resourcesMap[mountPath]
		if !ok {
			resource = &podagent.TResource{
				Id: makeIDFromMountPath(mountPath),
				DownloadMethod: &podagent.TResource_Files{
					Files: &podagent.TFiles{},
				},
				Verification: &podagent.TVerification{
					Disabled: true,
				},
			}
			resourcesMap[mountPath] = resource
		}
		resourceFiles := resource.GetFiles()
		resourceFiles.Files = append(resourceFiles.Files, podagentFile)
	}

	for _, resource := range resourcesMap {
		resourceFiles := resource.GetFiles().Files
		sort.Slice(resourceFiles, func(i, j int) bool {
			return resourceFiles[i].FileName < resourceFiles[j].FileName
		})
	}

	return resourcesMap, secretRefs
}

func makeURLStaticResources(urls map[string]string) map[string]*podagent.TResource {
	resources := make(map[string]*podagent.TResource, len(urls))

	for mountPath, url := range urls {
		resources[mountPath] = &podagent.TResource{
			Id: makeIDFromMountPath(mountPath),
			DownloadMethod: &podagent.TResource_Url{
				Url: url,
			},
			Verification: &podagent.TVerification{
				Disabled: true,
			},
		}
	}

	return resources
}

func makeLayers(layers []*v1proto.Spec_PortoLayer) []*podagent.TLayer {
	podagentLayers := make([]*podagent.TLayer, 0, len(layers))
	for _, layer := range layers {
		podagentLayers = append(podagentLayers, &podagent.TLayer{
			Id:       layer.Id,
			Checksum: "EMPTY:",
			DownloadMethod: &podagent.TLayer_Url{
				Url: fmt.Sprintf("sbr:%s", layer.ResourceId),
			},
		})
	}
	return podagentLayers
}

type resourcesResult struct {
	resources  *podagent.TResourceGang
	mounts     []*podagent.TMountedStaticResource
	secretRefs map[string]*ypapi.TSecretRef
}

func makeResources(
	files map[string]string,
	urls map[string]string,
	layers []*v1proto.Spec_PortoLayer,
) *resourcesResult {
	resourcesMap, secretRefs := makeFilesStaticResources(files)
	urlResourcesMap := makeURLStaticResources(urls)
	for mountPath, resource := range urlResourcesMap {
		resourcesMap[mountPath] = resource
	}

	mounts := make([]*podagent.TMountedStaticResource, 0, len(resourcesMap))
	for mountPath, resource := range resourcesMap {
		mounts = append(mounts, &podagent.TMountedStaticResource{
			ResourceRef: resource.Id,
			MountPoint:  mountPath,
		})
	}

	resources := []*podagent.TResource{}
	for _, resource := range resourcesMap {
		resources = append(resources, resource)
	}

	sort.Slice(resources, func(i, j int) bool {
		return resources[i].Id < resources[j].Id
	})
	sort.Slice(mounts, func(i, j int) bool {
		return mounts[i].ResourceRef < mounts[j].ResourceRef
	})

	return &resourcesResult{
		resources: &podagent.TResourceGang{
			StaticResources: resources,
			Layers:          makeLayers(layers),
		},
		mounts:     mounts,
		secretRefs: secretRefs,
	}
}
