package controllers

import (
	"testing"

	"github.com/stretchr/testify/require"

	v1proto "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

func TestMakeIDFromMountPath(t *testing.T) {
	t.Parallel()

	testCases := []struct {
		mountPath string
		id        string
	}{
		{"", "-"},
		{"/", "-"},
		{"/a/b", "a-b"},
		{"/a/b/c/", "a-b-c"},
		{"/a-b/c-d/", "a-b-c-d"},
	}

	for _, tc := range testCases {
		tc := tc
		t.Run(tc.mountPath, func(t *testing.T) {
			t.Parallel()
			id := makeIDFromMountPath(tc.mountPath)
			require.Equal(t, tc.id, id)
		})
	}
}

func TestMakeFilesStaticResources(t *testing.T) {
	t.Parallel()

	resources, refs := makeFilesStaticResources(
		map[string]string{
			"/mount1/plain":  "text1",
			"/mount1/secret": "${id:ver:key}",
			"/mount2/plain":  "text2",
		},
	)
	requireProtoEqual(
		t,
		map[string]*podagent.TResource{
			"/mount1/": {
				Id: "mount1",
				DownloadMethod: &podagent.TResource_Files{
					Files: &podagent.TFiles{
						Files: []*podagent.TFile{
							{
								FileName: "plain",
								Content:  &podagent.TFile_RawData{RawData: "text1"},
							},
							{
								FileName: "secret",
								Content: &podagent.TFile_SecretData{
									SecretData: &podagent.SecretSelector{
										Alias: "id:ver",
										Id:    "key",
									},
								},
							},
						},
					},
				},
				Verification: &podagent.TVerification{
					Disabled: true,
				},
			},
			"/mount2/": {
				Id: "mount2",
				DownloadMethod: &podagent.TResource_Files{
					Files: &podagent.TFiles{
						Files: []*podagent.TFile{
							{
								FileName: "plain",
								Content:  &podagent.TFile_RawData{RawData: "text2"},
							},
						},
					},
				},
				Verification: &podagent.TVerification{
					Disabled: true,
				},
			},
		},
		resources,
	)
	requireProtoEqual(
		t,
		map[string]*ypapi.TSecretRef{
			"id:ver": {
				SecretId:      ptr.String("id"),
				SecretVersion: ptr.String("ver"),
			},
		},
		refs,
	)
}

func TestMakeSandboxStaticResource(t *testing.T) {
	t.Parallel()

	resources := makeURLStaticResources(
		map[string]string{
			"/test1": "sbr:123",
			"/test2": "https://test.com/",
		})
	requireProtoEqual(
		t,
		map[string]*podagent.TResource{
			"/test1": {
				Id: "test1",
				DownloadMethod: &podagent.TResource_Url{
					Url: "sbr:123",
				},
				Verification: &podagent.TVerification{
					Disabled: true,
				},
			},
			"/test2": {
				Id: "test2",
				DownloadMethod: &podagent.TResource_Url{
					Url: "https://test.com/",
				},
				Verification: &podagent.TVerification{
					Disabled: true,
				},
			},
		},
		resources,
	)
}

func TestMakeLayers(t *testing.T) {
	t.Parallel()

	layers := makeLayers([]*v1proto.Spec_PortoLayer{
		{
			Id:         "layer1",
			ResourceId: "123",
		},
		{
			Id:         "layer2",
			ResourceId: "456",
		},
	})

	requireProtoEqual(
		t,
		[]*podagent.TLayer{
			{
				Id:             "layer1",
				Checksum:       "EMPTY:",
				DownloadMethod: &podagent.TLayer_Url{Url: "sbr:123"},
			},
			{
				Id:             "layer2",
				Checksum:       "EMPTY:",
				DownloadMethod: &podagent.TLayer_Url{Url: "sbr:456"},
			},
		},
		layers,
	)
}

func TestMakeResources(t *testing.T) {
	t.Parallel()

	result := makeResources(
		map[string]string{
			"/secmount/secret": "${id:ver:key}",
		},
		map[string]string{
			"/sandboxmount": "sbr:123",
		},
		[]*v1proto.Spec_PortoLayer{
			{Id: "layer", ResourceId: "456"},
		},
	)

	requireProtoEqual(
		t,
		&podagent.TResourceGang{
			StaticResources: []*podagent.TResource{
				{
					Id: "sandboxmount",
					DownloadMethod: &podagent.TResource_Url{
						Url: "sbr:123",
					},
					Verification: &podagent.TVerification{
						Disabled: true,
					},
				},
				{
					Id: "secmount",
					DownloadMethod: &podagent.TResource_Files{
						Files: &podagent.TFiles{
							Files: []*podagent.TFile{
								{
									FileName: "secret",
									Content: &podagent.TFile_SecretData{
										SecretData: &podagent.SecretSelector{
											Alias: "id:ver",
											Id:    "key",
										},
									},
								},
							},
						},
					},
					Verification: &podagent.TVerification{
						Disabled: true,
					},
				},
			},
			Layers: []*podagent.TLayer{
				{
					Id:             "layer",
					Checksum:       "EMPTY:",
					DownloadMethod: &podagent.TLayer_Url{Url: "sbr:456"},
				},
			},
		},
		result.resources,
	)

	requireProtoEqual(
		t,
		[]*podagent.TMountedStaticResource{
			{
				ResourceRef: "sandboxmount",
				MountPoint:  "/sandboxmount",
			},
			{
				ResourceRef: "secmount",
				MountPoint:  "/secmount/",
			},
		},
		result.mounts,
	)

	requireProtoEqual(
		t,
		map[string]*ypapi.TSecretRef{
			"id:ver": {
				SecretId:      ptr.String("id"),
				SecretVersion: ptr.String("ver"),
			},
		},
		result.secretRefs,
	)
}
