package controllers

import (
	"context"
	"fmt"
	"time"

	"github.com/go-logr/logr"
	"google.golang.org/protobuf/types/known/timestamppb"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/log"

	dsv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	prv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	sev1 "a.yandex-team.ru/infra/infractl/models/serviceendpoint/v1"
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

// RuntimeReconciler reconciles a Runtime object
type RuntimeReconciler struct {
	client.Client
	Scheme  *runtime.Scheme
	Stats   *Stats
	Options rv1.ReconcilerOptions
}

func (r *RuntimeReconciler) saveSyncError(
	ctx context.Context,
	log logr.Logger,
	kRuntime *rv1.Runtime,
	e error,
	msg string,
) {
	log.Error(e, msg)

	if kRuntime.Status.SyncStatus == nil {
		kRuntime.Status.SyncStatus = &prv1.SyncStatus{}
	}
	kRuntime.Status.SyncStatus.Error = &ypapi.TCondition{
		Status:             ypapi.EConditionStatus_CS_TRUE,
		Reason:             msg,
		Message:            e.Error(),
		LastTransitionTime: timestamppb.Now(),
	}
	kRuntime.Status.SyncStatus.Success = nil

	r.Stats.SyncErrors.Update(1)
	if e = r.Status().Update(ctx, kRuntime); e != nil {
		log.Error(e, "Failed to update status")
		log.Info(fmt.Sprintf("%T", e))
	}
}

func (r *RuntimeReconciler) saveSyncSuccess(
	ctx context.Context,
	log logr.Logger,
	kRuntime *rv1.Runtime,
) {
	log.V(3).Info("Saving sync success")
	if kRuntime.Status.SyncStatus == nil {
		kRuntime.Status.SyncStatus = &prv1.SyncStatus{}
	}
	kRuntime.Status.SyncStatus.Error = nil
	kRuntime.Status.SyncStatus.Success = &ypapi.TCondition{
		Status:             ypapi.EConditionStatus_CS_FALSE,
		Reason:             "Synced",
		LastTransitionTime: timestamppb.Now(),
	}

	if err := r.Status().Update(ctx, kRuntime); err != nil {
		log.Error(err, "Failed to update status")
		log.Info(fmt.Sprintf("%T", err))
	}
}

func updateRuntimeStatus(kRuntime *rv1.Runtime, kStage *dsv1.DeployStage) {
	if kRuntime.Status.SyncStatus == nil {
		kRuntime.Status.SyncStatus = &prv1.SyncStatus{}
	}
	if kRuntime.Status.SyncStatus.Stage == nil {
		kRuntime.Status.SyncStatus.Stage = &prv1.SyncStatus_StageStatus{}
	}
	kRuntime.Status.SyncStatus.Stage.ObservedGeneration = kRuntime.Generation
	kRuntime.Status.SyncStatus.Stage.DeploystageGeneration = kStage.Generation
}

func (r *RuntimeReconciler) processRuntime(
	ctx context.Context,
	log logr.Logger,
	kRuntime *rv1.Runtime,
) (syncStatus bool, kStage *dsv1.DeployStage, msg string, err error) {
	var kServiceEndpoint *sev1.ServiceEndpoint
	syncStatus = false
	needCreateStage := false
	needCreateServiceEndpoint := false
	needRemoveServiceEndpoint := false

	kStageName := makeName(rv1.NamespacedName(kRuntime), "stage")
	kServiceEndpointName := makeName(rv1.NamespacedName(kRuntime), "serviceendpoint")
	kStage, err = getObject[*dsv1.DeployStage](r, ctx, kRuntime, "stage")

	if err != nil {
		msg = "Failed to check stage existence"
		log.Error(err, msg, "stage", kStageName)
		return
	}

	if kStage == nil {
		if kStage, err = r.MakeEmptyStage(kRuntime); err != nil {
			msg = "Failed to set stage owner"
			log.Error(err, msg, "stage", kStageName)
			return
		}
		needCreateStage = true
	}

	kServiceEndpoint, err = getObject[*sev1.ServiceEndpoint](r, ctx, kRuntime, "serviceendpoint")
	if err != nil {
		msg = "Failed to check service endpoint existence"
		log.Error(err, msg, "serviceendpoint", kServiceEndpointName)
		return
	}

	if kServiceEndpoint == nil {
		if kServiceEndpoint, err = r.MakeEmptyServiceEndpoint(kRuntime); err != nil {
			msg = "Failed to set service endpoint owner"
			log.Error(err, msg, "serviceendpoint", kServiceEndpointName)
			return
		}
		needCreateServiceEndpoint = true
	}

	if !kRuntime.StageSpecNeedsUpdate(kStage) {
		return
	}

	builder := stageBuilder{r.Client}
	if err = builder.FillSpecs(ctx, kRuntime, kStage, kServiceEndpoint); err != nil {
		msg = "Failed to fill specs"
		log.Error(err, msg, "stage", kStageName, "serviceendpoint", kServiceEndpointName)
		return
	}

	if kServiceEndpoint.Spec == nil {
		if needCreateServiceEndpoint {
			needRemoveServiceEndpoint = true
		}
		needCreateServiceEndpoint = false
	}

	log.V(3).Info("Filled YP stage")

	if needCreateStage {
		err = r.Create(ctx, kStage)
	} else {
		err = r.Update(ctx, kStage)
	}
	if err != nil {
		msg = "Failed to update stage in k8s"
		log.Error(err, msg, "stage", kStageName)
		return
	} else {
		log.Info("Committed DeployStage spec", "stage", kStageName, "generation", kStage.Generation)
	}

	if needCreateServiceEndpoint {
		err = r.Create(ctx, kServiceEndpoint)
	} else if needRemoveServiceEndpoint {
		err = r.Delete(ctx, kServiceEndpoint)
	} else {
		err = r.Update(ctx, kServiceEndpoint)
	}
	if err != nil {
		msg = "Failed to update service endpoint in k8s"
		log.Error(err, msg, "serviceendpoint", kServiceEndpointName)
		return
	} else {
		log.Info("Committed ServiceEndpoint spec", "serviceendpoint", kStageName, "generation", kServiceEndpoint.Generation)
	}

	syncStatus = true

	return syncStatus, kStage, "", nil
}

func makeName(req types.NamespacedName, kind string) types.NamespacedName {
	return types.NamespacedName{
		Namespace: req.Namespace,
		Name:      kind + "-" + req.Name,
	}
}

type objectPtr[K any] interface {
	*K
	client.Object
}

func getObject[K objectPtr[U], U any](
	r *RuntimeReconciler,
	ctx context.Context,
	kRuntime *rv1.Runtime,
	kind string,
) (obj K, err error) {
	err = nil
	obj = new(U)

	name := makeName(rv1.NamespacedName(kRuntime), kind)
	if err = r.Get(ctx, name, obj); err != nil {
		var ret K
		obj = ret
		err = client.IgnoreNotFound(err)
	}
	return

}

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Runtime object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.8.3/pkg/reconcile
func (r *RuntimeReconciler) Reconcile(
	ctx context.Context,
	req ctrl.Request,
) (result ctrl.Result, err error) {
	defer unistat.MeasureMicrosecondsSince(r.Stats.SyncLatency, time.Now())
	defer r.Stats.SyncCount.Update(1)

	log := log.FromContext(ctx)
	log.Info("Syncing")
	defer log.Info("Sync finished")

	result = ctrl.Result{
		RequeueAfter: time.Second * 60,
	}

	kRuntime := &rv1.Runtime{}
	if err = r.Get(ctx, req.NamespacedName, kRuntime); err != nil {
		log.Error(err, "unable to fetch Runtime")
		err = client.IgnoreNotFound(err)
		if err == nil {
			result = ctrl.Result{}
		} else {
			r.Stats.SyncErrors.Update(1)
		}
		return
	}
	syncStatus, kStage, msg, err := r.processRuntime(ctx, log, kRuntime)
	if err != nil || syncStatus {
		// we need to reload object here, otherwise k8s would deny
		// updating status because of
		log.Info("Will reload runtime before updating status")
		if err2 := r.Get(ctx, req.NamespacedName, kRuntime); err2 != nil {
			log.Error(err2, "Failed to reload object before status update, will attempt anyway")
		}
	}

	updateRuntimeStatus(kRuntime, kStage)

	if kRuntime.Status.SyncStatus == nil {
		kRuntime.Status.SyncStatus = &prv1.SyncStatus{}
	}
	kRuntime.Status.SyncStatus.Url = kStage.MakeURL(r.Options.TemplateURL)
	if err != nil {
		log.Error(err, "Runtime processed with error", "msg", msg, "syncStatus", syncStatus)

		r.saveSyncError(ctx, log, kRuntime, err, msg)
		r.Stats.SyncErrors.Update(1)
		// we always return requeue-result, so we do not need additional controller logic
		// to trigger on error and requeue task immediately
		err = nil
	} else if syncStatus {
		log.Info("Runtime synced, will update status")
		r.saveSyncSuccess(ctx, log, kRuntime)
	}
	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *RuntimeReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&rv1.Runtime{}).
		Owns(&dsv1.DeployStage{}).
		Complete(r)
}
