package controllers

import (
	"context"
	"fmt"

	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"

	pdsv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/proto_v1"
	dsv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	sev1 "a.yandex-team.ru/infra/infractl/models/serviceendpoint/v1"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type stageBuilder struct {
	Client client.Client
}

func newStageSpec() *ypapi.TStageSpec {
	return &ypapi.TStageSpec{
		DeployUnits:      map[string]*ypapi.TDeployUnitSpec{},
		AccountId:        "tmp",
		DynamicResources: map[string]*ypapi.TStageSpec_TStageDynamicResourceSpec{},
		Env:              map[string]string{},
	}
}

func (r *RuntimeReconciler) MakeEmptyStage(kRuntime *rv1.Runtime) (kStage *dsv1.DeployStage, err error) {
	kStageName := makeName(rv1.NamespacedName(kRuntime), "stage")

	kStage = &dsv1.DeployStage{
		ObjectMeta: metav1.ObjectMeta{
			Name:        kStageName.Name,
			Namespace:   kStageName.Namespace,
			Generation:  1,
			Labels:      map[string]string{},
			Annotations: map[string]string{},
		},
		Spec: &pdsv1.Spec{
			StageSpec: newStageSpec(),
		},
		Status: &pdsv1.Status{},
	}
	if err = controllerutil.SetControllerReference(kRuntime, kStage, r.Scheme); err != nil {
		kStage = nil
	}
	return
}

func (b *stageBuilder) FillSpecs(ctx context.Context, kRuntime *rv1.Runtime, kStage *dsv1.DeployStage, kServiceEndpoint *sev1.ServiceEndpoint) error {
	duResult, err := b.makeDeployUnit(ctx, kRuntime)
	if err != nil {
		return fmt.Errorf("deploy unit generation failed: %w", err)
	}

	kServiceEndpoint.Spec = duResult.Provides

	kStage.Spec.StageSpec = &ypapi.TStageSpec{
		DeployUnits: map[string]*ypapi.TDeployUnitSpec{kRuntime.Name: duResult.DeployUnit},
	}
	return nil
}
