package controllers

import (
	"context"
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"google.golang.org/protobuf/testing/protocmp"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/client-go/kubernetes/scheme"

	dproto_v1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/proto_v1"
	dv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	"a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	v1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	sev1 "a.yandex-team.ru/infra/infractl/models/serviceendpoint/v1"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

func requireProtoEqual(t *testing.T, want, got any) {
	t.Helper()
	if diff := cmp.Diff(want, got, protocmp.Transform()); diff != "" {
		t.Errorf("not equal (-want +got):\n%s", diff)
	}
}

func TestFillStageSpec(t *testing.T) {
	kRuntime := &v1.Runtime{
		ObjectMeta: metav1.ObjectMeta{
			Name:        "runtime1",
			Namespace:   "ns1",
			Generation:  2,
			Labels:      map[string]string{},
			Annotations: map[string]string{},
		},
		Spec: &proto_v1.Spec{
			Image: "test/image:1.0@sha256:aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
			Compute: &proto_v1.Spec_ComputeResources{
				Vcpu:   "2000",
				Memory: "1G",
				Net:    "10M",
			},
			Logs:      &proto_v1.Spec_LogsTransmission{Enabled: proto_v1.Spec_LogsTransmission_LOGS_ENABLED},
			Readiness: nil,
			Liveness:  nil,
			Lifecycle: nil,
			Command:   "",
			// NOTE we intentionally test exactly one envvar here,
			// because we convert map->array and the order in array may be arbitrary
			Env: map[string]string{
				"a": "${sec-1:ver-2:value}",
			},
			Storage: &proto_v1.Spec_Storage{
				Quota:       "5G",
				IoBandwidth: "15M",
			},
			// Volumes:          nil,
			// StaticResources:  nil,
			Replicas: map[string]uint32{
				"sas": 1,
				"man": 2,
			},
			NetworkId: "_SEARCHSAND_",
		},
	}

	err := v1.AddToScheme(scheme.Scheme)
	assert.NoError(t, err, "CRD register failed")

	/*
			// Note: testenv start would always fail if only testenv binaries ain't installed,
		    // so we skip configuring cluster for now.
			// See https://book.kubebuilder.io/reference/envtest.html for more.

			testEnv := &envtest.Environment{
				CRDDirectoryPaths:     []string{filepath.Join("..", "api", "proto_v1")},
				ErrorIfCRDPathMissing: false,
			}
			cfg, err := testEnv.Start()
			assert.NoError(t, err, "TestEnv start failed")

			k8sClient, err := client.New(cfg, client.Options{Scheme: scheme.Scheme})
			assert.NoError(t, err, "Client create failed")
	*/

	reconciler := &RuntimeReconciler{
		Client:  nil,
		Scheme:  scheme.Scheme,
		Stats:   nil,
		Options: v1.ReconcilerOptions{TemplateURL: "https://man-pre.deploy.yandex-team.ru/%s/%s"},
	}
	kStage, err := reconciler.MakeEmptyStage(kRuntime)
	assert.NoError(t, err, "Stage creation failed")

	kServiceEndpoint := &sev1.ServiceEndpoint{}

	builder := stageBuilder{nil}
	err = builder.FillSpecs(context.TODO(), kRuntime, kStage, kServiceEndpoint)
	assert.NoError(t, err, "Stage fill failed")

	diskLabels, err := makeAttributesMap([]*Attribute{
		{
			Key:   "used_by_infra",
			Value: ptr.Bool(true),
		},
	})
	require.NoError(t, err, "make attributes failed")

	expectedStage := &dv1.DeployStage{
		Spec: &dproto_v1.Spec{
			StageSpec: &ypapi.TStageSpec{
				Revision: 0,
				DeployUnits: map[string]*ypapi.TDeployUnitSpec{
					"runtime1": {
						NetworkDefaults: &ypapi.TNetworkDefaults{NetworkId: "_SEARCHSAND_"}, // FIXME
						PodDeployPrimitive: &ypapi.TDeployUnitSpec_ReplicaSet{ReplicaSet: &ypapi.TDeployUnitSpec_TReplicaSetDeploy{
							ReplicaSetTemplate: &ypapi.TReplicaSetSpec{
								Constraints: &ypapi.TReplicaSetSpec_TConstraints{
									AntiaffinityConstraints: []*ypapi.TAntiaffinityConstraint{
										{Key: ptr.String("node"), MaxPods: ptr.Int64(1)},
									},
								},
								PodTemplateSpec: &ypapi.TPodTemplateSpec{
									Spec: &ypapi.TPodSpec{
										PodAgentPayload: &ypapi.TPodSpec_TPodAgentPayload{
											Spec: &podagent.TPodAgentSpec{
												Id:        "",
												Resources: &podagent.TResourceGang{},
												Volumes:   nil,
												Workloads: []*podagent.TWorkload{{
													Id:     "runtime1",
													BoxRef: "runtime1",
													Start:  &podagent.TUtilityContainer{},
													LivenessCheck: &podagent.TLivenessCheck{
														Backend: &podagent.TLivenessCheck_HttpGet{HttpGet: &podagent.THttpGet{
															Port: 80,
															Path: "/livez",
															TimeLimit: &podagent.TTimeLimit{
																InitialDelayMs:     5000,
																MinRestartPeriodMs: 3000,
																MaxRestartPeriodMs: 30000,
															},
														}},
													},
													ReadinessCheck: &podagent.TReadinessCheck{
														Backend: &podagent.TReadinessCheck_HttpGet{HttpGet: &podagent.THttpGet{
															Port: 80,
															Path: "/readyz",
															TimeLimit: &podagent.TTimeLimit{
																InitialDelayMs:     5000,
																MinRestartPeriodMs: 3000,
																MaxRestartPeriodMs: 30000,
															},
														}},
													},
													TransmitLogs: true,
												}},
												Boxes: []*podagent.TBox{{
													Id:               "runtime1",
													Rootfs:           &podagent.TRootfsVolume{},
													Volumes:          nil,
													Init:             nil,
													StaticResources:  nil,
													BindSkynet:       false,
													ComputeResources: nil,
													ResolvConf:       0,
													Env: []*podagent.TEnvVar{
														{
															Name: "a",
															Value: &podagent.TEnvVarValue{Value: &podagent.TEnvVarValue_SecretEnv{
																SecretEnv: &podagent.SecretSelector{
																	Alias: "sec-1:ver-2",
																	Id:    "value",
																},
															}},
														},
													},
													IsolationMode:     podagent.EContainerIsolationMode_EContainerIsolationMode_CHILD_ONLY,
													CgroupFsMountMode: podagent.ECgroupFsMountMode_ECgroupFsMountMode_RO,
												}},
												MutableWorkloads: []*podagent.TMutableWorkload{{WorkloadRef: "runtime1"}},
											},
										},
										ResourceRequests: &ypapi.TPodSpec_TResourceRequests{
											VcpuGuarantee:             ptr.Uint64(2000),
											VcpuLimit:                 ptr.Uint64(2000),
											MemoryGuarantee:           ptr.Uint64(1024 * 1024 * 1024),
											MemoryLimit:               ptr.Uint64(1024 * 1024 * 1024),
											NetworkBandwidthGuarantee: ptr.Uint64(10 * 1024 * 1024),
										},
										DiskVolumeRequests: []*ypapi.TPodSpec_TDiskVolumeRequest{{
											Id:           ptr.String("disk-0"),
											Labels:       diskLabels,
											StorageClass: ptr.String("hdd"),
											ConcretePolicy: &ypapi.TPodSpec_TDiskVolumeRequest_QuotaPolicy{QuotaPolicy: &ypapi.TPodSpec_TDiskVolumeRequest_TQuotaPolicy{
												Capacity:           ptr.Uint64(5 * 1024 * 1024 * 1024),
												BandwidthGuarantee: ptr.Uint64(15 * 1024 * 1024),
												BandwidthLimit:     ptr.Uint64(30 * 1024 * 1024),
											}},
										}},
										SecretRefs: map[string]*ypapi.TSecretRef{
											"sec-1:ver-2": {
												SecretId:      ptr.String("sec-1"),
												SecretVersion: ptr.String("ver-2"),
											},
										},
									},
								},
							},
							PerClusterSettings: map[string]*ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings{
								"sas": {
									PodCountPolicy: &ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings_PodCount{PodCount: 1},
									DeploymentStrategy: &ypapi.TReplicaSetSpec_TDeploymentStrategy{
										MaxUnavailable: 1,
									},
								},
								"man": {
									PodCountPolicy: &ypapi.TDeployUnitSpec_TReplicaSetDeploy_TPerClusterSettings_PodCount{PodCount: 2},
									DeploymentStrategy: &ypapi.TReplicaSetSpec_TDeploymentStrategy{
										MaxUnavailable: 1,
									},
								},
							},
						}},
						TvmConfig: nil,
						ImagesForBoxes: map[string]*ypapi.TDockerImageDescription{
							"runtime1": {
								RegistryHost: "registry.yandex.net",
								Name:         "test/image",
								Tag:          "1.0",
								Digest:       "sha256:aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa",
							},
						},
						EndpointSets:      nil,
						BoxJugglerConfigs: nil,
						LogbrokerConfig: &ypapi.TLogbrokerConfig{
							PodAdditionalResourcesRequest: &ypapi.TPodSpec_TResourceRequests{
								VcpuGuarantee: ptr.Uint64(0),
								VcpuLimit:     ptr.Uint64(0),
							},
						},
						LogrotateConfigs:                  nil,
						Revision:                          0,
						SoxService:                        false,
						CoredumpConfig:                    nil,
						TvmSandboxInfo:                    nil,
						PodAgentSandboxInfo:               nil,
						PodAgentLayerSandboxInfo:          nil,
						LogbrokerToolsSandboxInfo:         nil,
						DynamicResourceUpdaterSandboxInfo: nil,
						DeploySettings:                    &ypapi.TDeployUnitSpec_TDeploySettings{DeployStrategy: ypapi.TDeployUnitSpec_TDeploySettings_SEQUENTIAL},
						PatchersRevision:                  0,
						EnableDynamicResourceUpdater:      false,
						InfraComponents:                   nil,
						CollectPortometricsFromSidecars:   true,
					},
				},
				DynamicResources:                 map[string]*ypapi.TStageSpec_TStageDynamicResourceSpec{},
				RevisionInfo:                     nil,
				SoxService:                       false,
				ResourceCaches:                   nil,
				Env:                              map[string]string{},
				DisableAutomaticSecretDelegation: false,
				DeployUnitSettings:               nil,
			},
		},
	}

	requireProtoEqual(t, expectedStage.Spec.StageSpec, kStage.Spec.StageSpec)
}
