package controllers

import (
	"context"
	"fmt"
	"regexp"
	"sort"

	"google.golang.org/protobuf/proto"
	"google.golang.org/protobuf/types/known/anypb"
	"k8s.io/apimachinery/pkg/types"

	prv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	psev1 "a.yandex-team.ru/infra/infractl/models/serviceendpoint/proto_v1"
	sev1 "a.yandex-team.ru/infra/infractl/models/serviceendpoint/v1"
	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

var endpointReferenceRe = regexp.MustCompile(`^(?:([^/]*)/)?([^:]+)(?::(.+))?$`)

type makeTvmConfigResult struct {
	Config   *ypapi.TTvmConfig
	Secrets  []*deployutil.Secret
	Provides *psev1.Spec
}

func makeBlackboxEnvironment(env prv1.Spec_BlackboxEnvironment) string {
	switch env {
	case prv1.Spec_EBB_Prod:
		return "Prod"
	case prv1.Spec_EBB_Test:
		return "Test"
	case prv1.Spec_EBB_ProdYaTeam:
		return "ProdYaTeam"
	case prv1.Spec_EBB_TestYaTeam:
		return "TestYaTeam"
	case prv1.Spec_EBB_Stress:
		return "Stress"
	default:
		return "ProdYaTeam"
	}
}

func applyOverrides[T proto.Message](msg T, overridesAny *anypb.Any) (T, error) {
	if overridesAny == nil {
		return msg, nil
	}

	overrides, err := overridesAny.UnmarshalNew()
	if err != nil {
		var nilRet T
		return nilRet, err
	}

	if msg.ProtoReflect().Descriptor() != overrides.ProtoReflect().Descriptor() {
		var nilRet T
		return nilRet, fmt.Errorf(
			"override type %q mismatch api type %q",
			overrides.ProtoReflect().Descriptor().FullName(),
			msg.ProtoReflect().Descriptor().FullName(),
		)
	}
	proto.Merge(msg, overrides)
	return msg, nil
}

func parseReference(target *prv1.Spec_EndpointReference) (*prv1.Spec_UntypedObjectReference, error) {
	switch refType := target.GetRefType().(type) {
	case *prv1.Spec_EndpointReference_Endpoint:
		match := endpointReferenceRe.FindStringSubmatch(refType.Endpoint)
		if match == nil {
			return nil, fmt.Errorf("invalid object reference: %q", refType.Endpoint)
		}
		return &prv1.Spec_UntypedObjectReference{
			Namespace:   match[1],
			Name:        match[2],
			SubobjectId: match[3],
		}, nil
	case *prv1.Spec_EndpointReference_Struct:
		return refType.Struct, nil
	}
	return nil, fmt.Errorf("cannot parse endpoint reference")
}

func (b *stageBuilder) resolveConsumeReference(
	ctx context.Context,
	alias string,
	target *prv1.Spec_EndpointReference,
) (*ypapi.TTvmApp, error) {
	endpointTarget, err := parseReference(target)
	if err != nil {
		return nil, fmt.Errorf("invalid consume %q: %w", alias, err)
	}
	/*
		if len(endpointTarget.Namespace) == 0 {
			endpointTarget.Namespace = ???
		}
	*/
	name := types.NamespacedName{
		Namespace: endpointTarget.Namespace,
		Name:      endpointTarget.Name,
	}
	ep := &sev1.ServiceEndpoint{}
	err = b.Client.Get(ctx, name, ep)
	if err != nil {
		return nil, fmt.Errorf("failed to find ServiceEndpoint reference %v: %w", name.String(), err)
	}
	if len(endpointTarget.SubobjectId) == 0 && len(ep.GetSpec().GetEndpoints()) == 1 {
		for k := range ep.GetSpec().GetEndpoints() {
			endpointTarget.SubobjectId = k
			break
		}
	}
	subobject, ok := ep.GetSpec().GetEndpoints()[endpointTarget.SubobjectId]
	if !ok {
		return nil, fmt.Errorf("no api %q in ServiceEndpoint %v", endpointTarget.SubobjectId, name.String())
	}

	switch subobject := subobject.EndpointType.(type) {
	case *psev1.Endpoint_Tvm:
		tvmdata, err := applyOverrides(subobject.Tvm, target.Overrides)
		if err != nil {
			return nil, fmt.Errorf("cannot apply overrides to %v: %w", name.String(), err)
		}
		return &ypapi.TTvmApp{
			AppId: uint32(tvmdata.Id),
			Alias: alias,
		}, nil
	}
	return nil, nil
}

func (b *stageBuilder) makeTvmClient(
	sourceAlias string,
	tvm *prv1.Spec_Provide_RawTvm,
	destinations []*ypapi.TTvmApp,
) (*ypapi.TTvmClient, *deployutil.Secret, error) {
	secret, ok := deployutil.ParseSecret(tvm.Secret)
	if !ok {
		return nil, nil, fmt.Errorf("invalid secret given for TVM source %v: %q", tvm.Id, tvm.Secret)
	}

	client := &ypapi.TTvmClient{
		SecretSelector: &podagent.SecretSelector{
			Alias: secret.MakeAlias(),
			Id:    secret.Key,
		},
		Source: &ypapi.TTvmApp{
			AppId:        uint32(tvm.Id),
			Alias:        sourceAlias,
			AbcServiceId: "",
		},
		Destinations: destinations,
	}
	return client, secret, nil
}

func (b *stageBuilder) makeDestinations(
	ctx context.Context,
	targets map[string]*prv1.Spec_Consume,
) ([]*ypapi.TTvmApp, error) {
	var destinations []*ypapi.TTvmApp

	for alias, target := range targets {
		switch target := target.GetConsume().(type) {
		case *prv1.Spec_Consume_TvmId:
			destinations = append(destinations, &ypapi.TTvmApp{
				AppId: uint32(target.TvmId),
				Alias: alias,
			})
		case *prv1.Spec_Consume_Ref:
			destination, err := b.resolveConsumeReference(ctx, alias, target.Ref)
			if err != nil {
				return nil, err
			}
			if destination != nil {
				destinations = append(destinations, destination)
			}
		}
	}
	sort.Slice(destinations, func(i, j int) bool {
		return destinations[i].AppId < destinations[j].AppId
	})
	return destinations, nil
}

func (b *stageBuilder) makeTvmConfig(ctx context.Context, kRuntime *rv1.Runtime) (makeTvmConfigResult, error) {
	result := makeTvmConfigResult{}

	sources := kRuntime.GetSpec().GetProvides()

	if len(sources) == 0 {
		return result, nil
	}

	result.Provides = &psev1.Spec{
		Endpoints: map[string]*psev1.Endpoint{},
	}

	destinations, err := b.makeDestinations(ctx, kRuntime.GetSpec().GetConsumes())
	if err != nil {
		return makeTvmConfigResult{}, err
	}

	result.Config = &ypapi.TTvmConfig{
		Mode:                ypapi.TTvmConfig_DISABLED,
		BlackboxEnvironment: makeBlackboxEnvironment(kRuntime.GetSpec().GetBlackboxEnvironment()),
		Clients:             nil,
		ClientPort:          2,
	}

	for sourceID, source := range sources {
		switch provide := source.Provide.(type) {
		case *prv1.Spec_Provide_Tvm:
			result.Provides.Endpoints[sourceID] = &psev1.Endpoint{EndpointType: &psev1.Endpoint_Tvm{
				Tvm: &psev1.Tvm{Id: provide.Tvm.Id},
			}}
		}

		tvm := source.GetTvm()
		if tvm == nil {
			continue
		}
		client, secret, err := b.makeTvmClient(sourceID, tvm, destinations)
		if err != nil {
			return makeTvmConfigResult{}, err
		}
		if secret != nil {
			result.Secrets = append(result.Secrets, secret)
		}

		result.Config.Mode = ypapi.TTvmConfig_ENABLED
		result.Config.Clients = append(result.Config.Clients, client)
	}

	sort.Slice(result.Config.Clients, func(i, j int) bool {
		return result.Config.Clients[i].Source.AppId < result.Config.Clients[j].Source.AppId
	})
	sort.Slice(result.Secrets, func(i, j int) bool {
		a, b := result.Secrets[i], result.Secrets[j]
		return a.ID < b.ID || (a.ID == b.ID && (a.Version < b.Version || (a.Version == b.Version && a.Key < b.Key)))
	})

	return result, nil
}
