package controllers

import (
	"context"
	"encoding/json"
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
	"google.golang.org/protobuf/types/known/anypb"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/client-go/kubernetes/scheme"
	"sigs.k8s.io/controller-runtime/pkg/client"

	prv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	rv1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	psev1 "a.yandex-team.ru/infra/infractl/models/serviceendpoint/proto_v1"
	sev1 "a.yandex-team.ru/infra/infractl/models/serviceendpoint/v1"
	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

type copyableInto interface {
	client.Object
	DeepCopyInto(into copyableInto)
}
type fakeClient struct {
	client.Client
	Obj client.Object
}

func (c *fakeClient) Get(_ context.Context, _ types.NamespacedName, obj client.Object) error {
	if c.Obj == nil {
		return fmt.Errorf("no object found")
	}

	j, err := json.Marshal(c.Obj)
	if err != nil {
		return err
	}
	decoder := scheme.Codecs.UniversalDecoder()
	_, _, err = decoder.Decode(j, nil, obj)
	return err
}

func Test_makeTvmConfig(t *testing.T) {
	tests := []struct {
		name            string
		kRuntime        *rv1.Runtime
		serviceEndpoint *psev1.Spec
		wantConfig      *ypapi.TTvmConfig
		wantSecrets     []*deployutil.Secret
		wantProvides    *psev1.Spec
		wantErr         assert.ErrorAssertionFunc
	}{
		{
			name:         "No mesh, no tvm",
			kRuntime:     &rv1.Runtime{Spec: &prv1.Spec{}},
			wantProvides: nil,
			wantErr: func(t assert.TestingT, err error, i ...interface{}) bool {
				return assert.NoError(t, err, "makeTvmConfig(%v)", i[0])
			},
		},

		{
			name: "single TVM app with no consumes",
			kRuntime: &rv1.Runtime{Spec: &prv1.Spec{
				Provides: map[string]*prv1.Spec_Provide{
					"tvm12345": {
						Provide: &prv1.Spec_Provide_Tvm{
							Tvm: &prv1.Spec_Provide_RawTvm{Id: 12345, Secret: "${sec-1234:ver-2345:somekey}"},
						},
					},
				},
				Consumes:            nil,
				BlackboxEnvironment: prv1.Spec_EBB_Prod,
			}},
			wantConfig: &ypapi.TTvmConfig{
				Mode:                ypapi.TTvmConfig_ENABLED,
				BlackboxEnvironment: "Prod",
				Clients: []*ypapi.TTvmClient{
					{
						SecretSelector: &podagent.SecretSelector{Alias: "sec-1234:ver-2345", Id: "somekey"},
						Source:         &ypapi.TTvmApp{AppId: 12345, Alias: "tvm12345"},
						Destinations:   nil,
					},
				},
				ClientPort: 2,
			},
			wantSecrets: []*deployutil.Secret{{ID: "sec-1234", Version: "ver-2345", Key: "somekey"}},
			wantProvides: &psev1.Spec{Endpoints: map[string]*psev1.Endpoint{
				"tvm12345": {
					EndpointType: &psev1.Endpoint_Tvm{Tvm: &psev1.Tvm{Id: 12345}},
				},
			}},
			wantErr: func(t assert.TestingT, err error, i ...interface{}) bool {
				return assert.NoError(t, err, "makeTvmConfig(%v)", i[0])
			},
		},

		{
			name: "two TVM apps with two consumes",
			kRuntime: &rv1.Runtime{Spec: &prv1.Spec{
				Provides: map[string]*prv1.Spec_Provide{
					"tvm12345": {
						Provide: &prv1.Spec_Provide_Tvm{
							Tvm: &prv1.Spec_Provide_RawTvm{Id: 12345, Secret: "${sec-1234:ver-2345:somekey}"},
						},
					},
					"tvm23456": {
						Provide: &prv1.Spec_Provide_Tvm{
							Tvm: &prv1.Spec_Provide_RawTvm{Id: 23456, Secret: "${sec-3456:ver-4567:somekey2}"},
						},
					},
				},
				Consumes: map[string]*prv1.Spec_Consume{
					"tvm1": {
						Consume: &prv1.Spec_Consume_TvmId{TvmId: 77777},
					},
					"tvm2": {
						Consume: &prv1.Spec_Consume_TvmId{TvmId: 88888},
					},
				},
				BlackboxEnvironment: prv1.Spec_EBB_Prod,
			}},
			wantConfig: &ypapi.TTvmConfig{
				Mode:                ypapi.TTvmConfig_ENABLED,
				BlackboxEnvironment: "Prod",
				Clients: []*ypapi.TTvmClient{
					{
						SecretSelector: &podagent.SecretSelector{Alias: "sec-1234:ver-2345", Id: "somekey"},
						Source:         &ypapi.TTvmApp{AppId: 12345, Alias: "tvm12345"},
						Destinations: []*ypapi.TTvmApp{
							{AppId: 77777, Alias: "tvm1"},
							{AppId: 88888, Alias: "tvm2"},
						},
					},
					{
						SecretSelector: &podagent.SecretSelector{Alias: "sec-3456:ver-4567", Id: "somekey2"},
						Source:         &ypapi.TTvmApp{AppId: 23456, Alias: "tvm23456"},
						Destinations: []*ypapi.TTvmApp{
							{AppId: 77777, Alias: "tvm1"},
							{AppId: 88888, Alias: "tvm2"},
						},
					},
				},
				ClientPort: 2,
			},
			wantSecrets: []*deployutil.Secret{
				{ID: "sec-1234", Version: "ver-2345", Key: "somekey"},
				{ID: "sec-3456", Version: "ver-4567", Key: "somekey2"},
			},
			wantProvides: &psev1.Spec{Endpoints: map[string]*psev1.Endpoint{
				"tvm12345": {EndpointType: &psev1.Endpoint_Tvm{Tvm: &psev1.Tvm{Id: 12345}}},
				"tvm23456": {EndpointType: &psev1.Endpoint_Tvm{Tvm: &psev1.Tvm{Id: 23456}}},
			}},
			wantErr: func(t assert.TestingT, err error, i ...interface{}) bool {
				return assert.NoError(t, err, "makeTvmConfig(%v)", i[0])
			},
		},

		{
			name: "TVM app with reference TVM consume",
			kRuntime: &rv1.Runtime{Spec: &prv1.Spec{
				Provides: map[string]*prv1.Spec_Provide{
					"tvm12345": {
						Provide: &prv1.Spec_Provide_Tvm{
							Tvm: &prv1.Spec_Provide_RawTvm{Id: 12345, Secret: "${sec-1234:ver-2345:somekey}"},
						},
					},
				},
				Consumes: map[string]*prv1.Spec_Consume{
					"myapi": {
						Consume: &prv1.Spec_Consume_Ref{Ref: &prv1.Spec_EndpointReference{
							RefType: &prv1.Spec_EndpointReference_Struct{Struct: &prv1.Spec_UntypedObjectReference{SubobjectId: "myapi"}},
						}},
					},
				},
			}},
			serviceEndpoint: &psev1.Spec{Endpoints: map[string]*psev1.Endpoint{
				"myapi": {EndpointType: &psev1.Endpoint_Tvm{Tvm: &psev1.Tvm{Id: 77777}}},
			}},
			wantConfig: &ypapi.TTvmConfig{
				Mode:                ypapi.TTvmConfig_ENABLED,
				BlackboxEnvironment: "ProdYaTeam",
				Clients: []*ypapi.TTvmClient{
					{
						SecretSelector: &podagent.SecretSelector{Alias: "sec-1234:ver-2345", Id: "somekey"},
						Source:         &ypapi.TTvmApp{AppId: 12345, Alias: "tvm12345"},
						Destinations: []*ypapi.TTvmApp{
							{AppId: 77777, Alias: "myapi"},
						},
					},
				},
				ClientPort: 2,
			},
			wantSecrets: []*deployutil.Secret{
				{ID: "sec-1234", Version: "ver-2345", Key: "somekey"},
			},
			wantProvides: &psev1.Spec{Endpoints: map[string]*psev1.Endpoint{
				"tvm12345": {EndpointType: &psev1.Endpoint_Tvm{Tvm: &psev1.Tvm{Id: 12345}}},
			}},
			wantErr: func(t assert.TestingT, err error, i ...interface{}) bool {
				return assert.NoError(t, err, "makeTvmConfig(%v)", i[0])
			},
		},

		{
			name: "TVM app with reference TVM consume and override",
			kRuntime: &rv1.Runtime{Spec: &prv1.Spec{
				Provides: map[string]*prv1.Spec_Provide{
					"tvm12345": {
						Provide: &prv1.Spec_Provide_Tvm{
							Tvm: &prv1.Spec_Provide_RawTvm{Id: 12345, Secret: "${sec-1234:ver-2345:somekey}"},
						},
					},
				},
				Consumes: map[string]*prv1.Spec_Consume{
					"myapi": {
						Consume: &prv1.Spec_Consume_Ref{Ref: &prv1.Spec_EndpointReference{
							RefType: &prv1.Spec_EndpointReference_Struct{Struct: &prv1.Spec_UntypedObjectReference{
								SubobjectId: "myapi",
							}},
							Overrides: func() *anypb.Any {
								ret, _ := anypb.New(&psev1.Tvm{Id: 88888})
								return ret
							}(),
						}},
					},
				},
			}},
			serviceEndpoint: &psev1.Spec{Endpoints: map[string]*psev1.Endpoint{
				"myapi": {EndpointType: &psev1.Endpoint_Tvm{Tvm: &psev1.Tvm{Id: 77777}}},
			}},
			wantConfig: &ypapi.TTvmConfig{
				Mode:                ypapi.TTvmConfig_ENABLED,
				BlackboxEnvironment: "ProdYaTeam",
				Clients: []*ypapi.TTvmClient{
					{
						SecretSelector: &podagent.SecretSelector{Alias: "sec-1234:ver-2345", Id: "somekey"},
						Source:         &ypapi.TTvmApp{AppId: 12345, Alias: "tvm12345"},
						Destinations: []*ypapi.TTvmApp{
							{AppId: 88888, Alias: "myapi"},
						},
					},
				},
				ClientPort: 2,
			},
			wantSecrets: []*deployutil.Secret{
				{ID: "sec-1234", Version: "ver-2345", Key: "somekey"},
			},
			wantProvides: &psev1.Spec{Endpoints: map[string]*psev1.Endpoint{
				"tvm12345": {EndpointType: &psev1.Endpoint_Tvm{Tvm: &psev1.Tvm{Id: 12345}}},
			}},
			wantErr: func(t assert.TestingT, err error, i ...interface{}) bool {
				return assert.NoError(t, err, "makeTvmConfig(%v)", i[0])
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			ctx := context.Background()
			var se *sev1.ServiceEndpoint
			if tt.serviceEndpoint != nil {
				se = &sev1.ServiceEndpoint{Spec: tt.serviceEndpoint}
			}
			client := &fakeClient{nil, se}
			builder := &stageBuilder{client}
			gotResult, err := builder.makeTvmConfig(ctx, tt.kRuntime)
			if !tt.wantErr(t, err, fmt.Sprintf("makeTvmConfig(%v)", tt.kRuntime)) {
				return
			}

			assert.Equalf(t, tt.wantSecrets, gotResult.Secrets, "makeTvmConfig(%v)", tt.kRuntime)

			requireProtoEqual(t, tt.wantConfig, gotResult.Config)

			if tt.wantProvides == nil {
				assert.Nil(t, gotResult.Provides, "makeTvmConfig(%v)", tt.kRuntime)
			} else {
				if assert.NotNil(t, gotResult.Provides, "makeTvmConfig(%v)", tt.kRuntime) {
					requireProtoEqual(t, tt.wantProvides, gotResult.Provides)
				}
			}
		})
	}
}

func Test_parseReference(t *testing.T) {
	noError := func(t assert.TestingT, err error, i ...interface{}) bool {
		return assert.NoError(t, err, "parseReference(%v)", i[0])
	}
	tests := []struct {
		name    string
		target  *prv1.Spec_EndpointReference
		want    *prv1.Spec_UntypedObjectReference
		wantErr assert.ErrorAssertionFunc
	}{
		{
			name:    "just endpoint id",
			target:  &prv1.Spec_EndpointReference{RefType: &prv1.Spec_EndpointReference_Endpoint{Endpoint: "some_endpoint"}},
			want:    &prv1.Spec_UntypedObjectReference{Name: "some_endpoint"},
			wantErr: noError,
		},

		{
			name:    "endpoint with namespace",
			target:  &prv1.Spec_EndpointReference{RefType: &prv1.Spec_EndpointReference_Endpoint{Endpoint: "my_namespace/some_endpoint"}},
			want:    &prv1.Spec_UntypedObjectReference{Namespace: "my_namespace", Name: "some_endpoint"},
			wantErr: noError,
		},

		{
			name:    "endpoint with subobject id",
			target:  &prv1.Spec_EndpointReference{RefType: &prv1.Spec_EndpointReference_Endpoint{Endpoint: "some_endpoint:some_subobject"}},
			want:    &prv1.Spec_UntypedObjectReference{Name: "some_endpoint", SubobjectId: "some_subobject"},
			wantErr: noError,
		},

		{
			name:    "endpoint with namespace and subobject id",
			target:  &prv1.Spec_EndpointReference{RefType: &prv1.Spec_EndpointReference_Endpoint{Endpoint: "my_namespace/some_endpoint:some_subobject"}},
			want:    &prv1.Spec_UntypedObjectReference{Namespace: "my_namespace", Name: "some_endpoint", SubobjectId: "some_subobject"},
			wantErr: noError,
		},

		{
			name: "structured reference",
			target: &prv1.Spec_EndpointReference{RefType: &prv1.Spec_EndpointReference_Struct{Struct: &prv1.Spec_UntypedObjectReference{
				Namespace:   "my_namespace",
				Name:        "some_endpoint",
				SubobjectId: "some_subobject",
			}}},
			want:    &prv1.Spec_UntypedObjectReference{Namespace: "my_namespace", Name: "some_endpoint", SubobjectId: "some_subobject"},
			wantErr: noError,
		},

		{
			name:   "no reference",
			target: &prv1.Spec_EndpointReference{},
			want:   nil,
			wantErr: func(t assert.TestingT, err error, i ...interface{}) bool {
				return assert.Error(t, err, "parseReference(%v)", i[0])
			},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := parseReference(tt.target)
			if !tt.wantErr(t, err, fmt.Sprintf("parseReference(%v)", tt.target)) {
				return
			}
			requireProtoEqual(t, tt.want, got)
		})
	}
}
