package controllers

import (
	"fmt"
	"sort"

	v1proto "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	deployinfrav1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/infra/infractl/internal/quantities"
	"a.yandex-team.ru/infra/infractl/util/deployutil"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

var (
	DefaultReadinessCheck = &podagent.TReadinessCheck{
		Backend: &podagent.TReadinessCheck_HttpGet{
			HttpGet: &podagent.THttpGet{
				Port: 80,
				Path: "/readyz",
				TimeLimit: &podagent.TTimeLimit{
					InitialDelayMs:     5000,
					MaxRestartPeriodMs: 30000,
					MinRestartPeriodMs: 3000,
				},
			},
		},
	}
	DefaultLivenessCheck = &podagent.TLivenessCheck{
		Backend: &podagent.TLivenessCheck_HttpGet{
			HttpGet: &podagent.THttpGet{
				Port: 80,
				Path: "/livez",
				TimeLimit: &podagent.TTimeLimit{
					InitialDelayMs:     5000,
					MaxRestartPeriodMs: 30000,
					MinRestartPeriodMs: 3000,
				},
			},
		},
	}
)

func makeContainerSpec(spec *v1proto.Spec_Command) (*podagent.TUtilityContainer, error) {
	if spec == nil {
		return nil, nil
	}
	vcpu, err := quantities.ConvertFromHuman(spec.GetResources().GetVcpu())
	if err != nil {
		return nil, fmt.Errorf("invalid vcpu: %w", err)
	}
	memory, err := quantities.ConvertFromHuman(spec.GetResources().GetMemory())
	if err != nil {
		return nil, fmt.Errorf("invalid memory: %w", err)
	}
	container := &podagent.TUtilityContainer{
		ComputeResources: &podagent.TComputeResources{
			VcpuLimit:   vcpu,
			MemoryLimit: memory,
		},
		CommandLine: spec.Cmd,
		Cwd:         spec.Cwd,
	}
	container.ComputeResources.VcpuGuarantee = container.ComputeResources.VcpuLimit
	container.ComputeResources.MemoryGuarantee = container.ComputeResources.MemoryLimit
	return container, nil
}

func makeReadinessCheck(spec *v1proto.Spec_Probe) (*podagent.TReadinessCheck, error) {
	// TODO (torkve) failure_threshold and successful_threshold

	switch s := spec.GetProbe().(type) {
	case *v1proto.Spec_Probe_Command:
		container, err := makeContainerSpec(s.Command)
		if err != nil {
			return nil, fmt.Errorf("invalid command: %w", err)
		}
		return &podagent.TReadinessCheck{
			Backend: &podagent.TReadinessCheck_Container{
				Container: container,
			},
		}, nil
	case *v1proto.Spec_Probe_Http:
		return &podagent.TReadinessCheck{
			Backend: &podagent.TReadinessCheck_HttpGet{
				HttpGet: s.Http,
			},
		}, nil
	case *v1proto.Spec_Probe_Tcp:
		return &podagent.TReadinessCheck{
			Backend: &podagent.TReadinessCheck_TcpCheck{
				TcpCheck: s.Tcp,
			},
		}, nil
	}
	return DefaultReadinessCheck.DeepCopy(), nil
}

func makeLivenessCheck(spec *v1proto.Spec_Probe) (*podagent.TLivenessCheck, error) {
	// TODO (torkve) failure_threshold and successful_threshold

	switch spec.GetProbe().(type) {
	case *v1proto.Spec_Probe_Command:
		s := spec.Probe.(*v1proto.Spec_Probe_Command)
		container, err := makeContainerSpec(s.Command)
		if err != nil {
			return nil, fmt.Errorf("invalid command: %w", err)
		}
		return &podagent.TLivenessCheck{
			Backend: &podagent.TLivenessCheck_Container{
				Container: container,
			},
		}, nil
	case *v1proto.Spec_Probe_Http:
		s := spec.Probe.(*v1proto.Spec_Probe_Http)
		return &podagent.TLivenessCheck{
			Backend: &podagent.TLivenessCheck_HttpGet{
				HttpGet: s.Http,
			},
		}, nil
	case *v1proto.Spec_Probe_Tcp:
		s := spec.Probe.(*v1proto.Spec_Probe_Tcp)
		return &podagent.TLivenessCheck{
			Backend: &podagent.TLivenessCheck_TcpCheck{
				TcpCheck: s.Tcp,
			},
		}, nil
	}
	return DefaultLivenessCheck.DeepCopy(), nil
}

func makeStopPolicy(spec *v1proto.Spec_Action) (*podagent.TStopPolicy, error) {
	switch spec.GetAction().(type) {
	case *v1proto.Spec_Action_Command:
		s := spec.Action.(*v1proto.Spec_Action_Command)
		container, err := makeContainerSpec(s.Command)
		if err != nil {
			return nil, fmt.Errorf("invalid command: %w", err)
		}
		return &podagent.TStopPolicy{
			Backend: &podagent.TStopPolicy_Container{
				Container: container,
			},
			MaxTries: spec.MaxTries,
		}, nil
	case *v1proto.Spec_Action_Http:
		s := spec.Action.(*v1proto.Spec_Action_Http)
		return &podagent.TStopPolicy{
			Backend: &podagent.TStopPolicy_HttpGet{
				HttpGet: s.Http,
			},
			MaxTries: spec.MaxTries,
		}, nil
	case *v1proto.Spec_Action_Signal:
		s := spec.Action.(*v1proto.Spec_Action_Signal)
		return &podagent.TStopPolicy{
			Backend: &podagent.TStopPolicy_UnixSignal{
				UnixSignal: s.Signal,
			},
			MaxTries: spec.MaxTries,
		}, nil
	}

	return nil, nil
}

func makeDestroyPolicy(spec *v1proto.Spec_Action) (*podagent.TDestroyPolicy, error) {
	switch spec.GetAction().(type) {
	case *v1proto.Spec_Action_Command:
		s := spec.Action.(*v1proto.Spec_Action_Command)
		container, err := makeContainerSpec(s.Command)
		if err != nil {
			return nil, fmt.Errorf("invalid command: %w", err)
		}
		return &podagent.TDestroyPolicy{
			Backend: &podagent.TDestroyPolicy_Container{
				Container: container,
			},
			MaxTries: spec.MaxTries,
		}, nil
	case *v1proto.Spec_Action_Http:
		s := spec.Action.(*v1proto.Spec_Action_Http)
		return &podagent.TDestroyPolicy{
			Backend: &podagent.TDestroyPolicy_HttpGet{
				HttpGet: s.Http,
			},
			MaxTries: spec.MaxTries,
		}, nil
	case *v1proto.Spec_Action_Signal:
		// TODO forbid it during validation
		return nil, nil
	}
	return nil, nil
}

func makeChecks(
	readiness *v1proto.Spec_Probe,
	liveness *v1proto.Spec_Probe,
) (*podagent.TReadinessCheck, *podagent.TLivenessCheck, error) {
	readinessCheck, err := makeReadinessCheck(readiness)
	if err != nil {
		return nil, nil, fmt.Errorf("invalid readiness check: %w", err)
	}
	livenessCheck, err := makeLivenessCheck(liveness)
	if err != nil {
		return nil, nil, fmt.Errorf("invalid liveness check: %w", err)
	}
	return readinessCheck, livenessCheck, nil
}

func makePolicies(
	stop *v1proto.Spec_Action,
	destroy *v1proto.Spec_Action,
) (*podagent.TStopPolicy, *podagent.TDestroyPolicy, error) {
	stopPolicy, err := makeStopPolicy(stop)
	if err != nil {
		return nil, nil, fmt.Errorf("invalid stop policy: %w", err)
	}
	destroyPolicy, err := makeDestroyPolicy(destroy)
	if err != nil {
		return nil, nil, fmt.Errorf("invalid destroy policy: %w", err)
	}

	return stopPolicy, destroyPolicy, nil
}

func logsEnabled(logs *v1proto.Spec_LogsTransmission) bool {
	switch logs.GetEnabled() {
	case v1proto.Spec_LogsTransmission_LOGS_ENABLED:
		return true
	case v1proto.Spec_LogsTransmission_LOGS_DISABLED:
		return false
	default:
		return true
	}
}

func makeWorkload(
	workload *v1proto.Spec_Workload,
	id, boxRef string,
) (*podagent.TWorkload, map[string]*ypapi.TSecretRef, error) {
	envVars, refs := deployutil.ParseEnvValues(workload.GetEnv(), true)

	init := make([]*podagent.TUtilityContainer, 0, len(workload.GetInit()))
	for i, rInit := range workload.GetInit() {
		container, err := makeContainerSpec(rInit)
		if err != nil {
			return nil, nil, fmt.Errorf("invalid %d init in %s: %w", i, id, err)
		}
		init = append(init, container)
	}

	start, err := makeContainerSpec(workload.GetCommand())
	if err != nil {
		return nil, nil, fmt.Errorf("invalid command in %s: %w", id, err)
	}
	readiness, liveness, err := makeChecks(
		workload.GetReadiness(), workload.GetLiveness(),
	)
	if err != nil {
		return nil, nil, err
	}
	stop, destroy, err := makePolicies(
		workload.GetActions().GetStop(),
		workload.GetActions().GetDestroy(),
	)
	if err != nil {
		return nil, nil, err
	}
	return &podagent.TWorkload{
		Id:             id,
		BoxRef:         boxRef,
		Env:            envVars,
		Init:           init,
		Start:          start,
		ReadinessCheck: readiness,
		LivenessCheck:  liveness,
		StopPolicy:     stop,
		DestroyPolicy:  destroy,
		TransmitLogs:   logsEnabled(workload.GetLogs()),
	}, refs, nil
}

func makeWorkloads(
	kRuntime *deployinfrav1.Runtime,
) ([]*podagent.TWorkload, map[string]*ypapi.TSecretRef, error) {
	if len(kRuntime.Spec.Workloads) == 0 {
		readiness, liveness, err := makeChecks(kRuntime.Spec.Readiness, kRuntime.Spec.Liveness)
		if err != nil {
			return nil, nil, err
		}
		stop, destroy, err := makePolicies(
			kRuntime.Spec.GetLifecycle().GetStop(),
			kRuntime.Spec.GetLifecycle().GetDestroy(),
		)
		if err != nil {
			return nil, nil, err
		}

		return []*podagent.TWorkload{
			{
				Id:             kRuntime.Name,
				BoxRef:         kRuntime.Name,
				Env:            []*podagent.TEnvVar{},
				Start:          &podagent.TUtilityContainer{CommandLine: kRuntime.Spec.Command},
				ReadinessCheck: readiness,
				LivenessCheck:  liveness,
				StopPolicy:     stop,
				DestroyPolicy:  destroy,
				TransmitLogs:   logsEnabled(kRuntime.Spec.GetLogs()),
			},
		}, nil, nil
	}

	if len(kRuntime.Spec.Command) != 0 || kRuntime.Spec.Lifecycle != nil {
		return nil, nil, fmt.Errorf("'command' and 'lifecycle' must be empty when using workloads")
	}

	workloads := make([]*podagent.TWorkload, 0, len(kRuntime.Spec.Workloads))
	secretRefs := map[string]*ypapi.TSecretRef{}

	for id, workload := range kRuntime.Spec.Workloads {
		podagentWorkload, refs, err := makeWorkload(workload, id, kRuntime.Name)
		if err != nil {
			return nil, nil, fmt.Errorf("invalid workload %s: %w", id, err)
		}
		for alias, ref := range refs {
			secretRefs[alias] = ref
		}
		workloads = append(workloads, podagentWorkload)
	}

	sort.Slice(workloads, func(i, j int) bool {
		return workloads[i].Id < workloads[j].Id
	})

	return workloads, secretRefs, nil
}
