package controllers

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	v1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	v1proto "a.yandex-team.ru/infra/infractl/controllers/runtime/api/proto_v1"
	deployinfrav1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/library/go/ptr"
	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

func TestMakeContainerSpec(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name string
			cmd  *v1proto.Spec_Command
		}{
			{
				name: "invalid vcpu",
				cmd: &v1proto.Spec_Command{
					Resources: &v1proto.Spec_ComputeResources{Vcpu: "one million"},
				},
			},
			{
				name: "invalid memory",
				cmd: &v1proto.Spec_Command{
					Resources: &v1proto.Spec_ComputeResources{Memory: "one million"},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				_, err := makeContainerSpec(tc.cmd)
				require.Error(t, err)
			})
		}
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name   string
			cmd    *v1proto.Spec_Command
			result *podagent.TUtilityContainer
		}{
			{
				name:   "nil cmd",
				cmd:    nil,
				result: nil,
			},
			{
				name: "empty cmd",
				cmd:  &v1proto.Spec_Command{},
				result: &podagent.TUtilityContainer{
					ComputeResources: &podagent.TComputeResources{},
				},
			},
			{
				name: "full cmd",
				cmd: &v1proto.Spec_Command{
					Cmd: "cmd",
					Cwd: "cwd",
					Resources: &v1proto.Spec_ComputeResources{
						Vcpu:   "2000",
						Memory: "1G",
						Net:    "10M",
					},
				},
				result: &podagent.TUtilityContainer{
					ComputeResources: &podagent.TComputeResources{
						VcpuLimit:       2000,
						VcpuGuarantee:   2000,
						MemoryLimit:     1024 * 1024 * 1024,
						MemoryGuarantee: 1024 * 1024 * 1024,
					},
					CommandLine: "cmd",
					Cwd:         "cwd",
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				result, err := makeContainerSpec(tc.cmd)
				require.NoError(t, err)
				requireProtoEqual(t, tc.result, result)
			})
		}
	})
}

func TestMakeReadinessCheck(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		_, err := makeReadinessCheck(&v1proto.Spec_Probe{
			Probe: &v1proto.Spec_Probe_Command{
				Command: &v1proto.Spec_Command{
					Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
				},
			},
		})
		require.Error(t, err)
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name   string
			probe  *v1proto.Spec_Probe
			result *podagent.TReadinessCheck
		}{
			{
				name:   "nil probe",
				probe:  nil,
				result: DefaultReadinessCheck,
			},
			{
				name:   "empty probe",
				probe:  &v1proto.Spec_Probe{},
				result: DefaultReadinessCheck,
			},
			{
				name: "command",
				probe: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Command{
						Command: &v1proto.Spec_Command{
							Cmd: "cmd",
						},
					},
				},
				result: &podagent.TReadinessCheck{
					Backend: &podagent.TReadinessCheck_Container{
						Container: &podagent.TUtilityContainer{
							ComputeResources: &podagent.TComputeResources{},
							CommandLine:      "cmd",
						},
					},
				},
			},
			{
				name: "http",
				probe: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Http{
						Http: &podagent.THttpGet{Port: 42},
					},
				},
				result: &podagent.TReadinessCheck{
					Backend: &podagent.TReadinessCheck_HttpGet{
						HttpGet: &podagent.THttpGet{Port: 42},
					},
				},
			},
			{
				name: "tcp",
				probe: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Tcp{
						Tcp: &podagent.TTcpCheck{Port: 42},
					},
				},
				result: &podagent.TReadinessCheck{
					Backend: &podagent.TReadinessCheck_TcpCheck{
						TcpCheck: &podagent.TTcpCheck{Port: 42},
					},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				result, err := makeReadinessCheck(tc.probe)
				require.NoError(t, err)
				requireProtoEqual(t, tc.result, result)
			})
		}
	})
}

func TestMakeLivenessCheck(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		_, err := makeLivenessCheck(&v1proto.Spec_Probe{
			Probe: &v1proto.Spec_Probe_Command{
				Command: &v1proto.Spec_Command{
					Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
				},
			},
		})
		require.Error(t, err)
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name   string
			probe  *v1proto.Spec_Probe
			result *podagent.TLivenessCheck
		}{
			{
				name:   "nil probe",
				probe:  nil,
				result: DefaultLivenessCheck,
			},
			{
				name:   "empty probe",
				probe:  &v1proto.Spec_Probe{},
				result: DefaultLivenessCheck,
			},
			{
				name: "command",
				probe: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Command{
						Command: &v1proto.Spec_Command{
							Cmd: "cmd",
						},
					},
				},
				result: &podagent.TLivenessCheck{
					Backend: &podagent.TLivenessCheck_Container{
						Container: &podagent.TUtilityContainer{
							ComputeResources: &podagent.TComputeResources{},
							CommandLine:      "cmd",
						},
					},
				},
			},
			{
				name: "http",
				probe: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Http{
						Http: &podagent.THttpGet{Port: 42},
					},
				},
				result: &podagent.TLivenessCheck{
					Backend: &podagent.TLivenessCheck_HttpGet{
						HttpGet: &podagent.THttpGet{Port: 42},
					},
				},
			},
			{
				name: "tcp",
				probe: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Tcp{
						Tcp: &podagent.TTcpCheck{Port: 42},
					},
				},
				result: &podagent.TLivenessCheck{
					Backend: &podagent.TLivenessCheck_TcpCheck{
						TcpCheck: &podagent.TTcpCheck{Port: 42},
					},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				result, err := makeLivenessCheck(tc.probe)
				require.NoError(t, err)
				requireProtoEqual(t, tc.result, result)
			})
		}
	})
}

func TestMakeStopPolicy(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		_, err := makeStopPolicy(&v1proto.Spec_Action{
			Action: &v1proto.Spec_Action_Command{
				Command: &v1proto.Spec_Command{
					Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
				},
			},
		})
		require.Error(t, err)
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name   string
			action *v1proto.Spec_Action
			result *podagent.TStopPolicy
		}{
			{
				name:   "nil action",
				action: nil,
				result: nil,
			},
			{
				name:   "empty action",
				action: &v1proto.Spec_Action{},
				result: nil,
			},
			{
				name: "command",
				action: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Command{},
				},
				result: &podagent.TStopPolicy{
					Backend: &podagent.TStopPolicy_Container{},
				},
			},
			{
				name: "http",
				action: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Http{
						Http: &podagent.THttpGet{Port: 42},
					},
				},
				result: &podagent.TStopPolicy{
					Backend: &podagent.TStopPolicy_HttpGet{
						HttpGet: &podagent.THttpGet{Port: 42},
					},
				},
			},
			{
				name: "signal",
				action: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Signal{
						Signal: &podagent.TUnixSignal{
							Signal: podagent.EUnixSignalType_EUnixSignalType_SIGALRM,
						},
					},
				},
				result: &podagent.TStopPolicy{
					Backend: &podagent.TStopPolicy_UnixSignal{
						UnixSignal: &podagent.TUnixSignal{
							Signal: podagent.EUnixSignalType_EUnixSignalType_SIGALRM,
						},
					},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				result, err := makeStopPolicy(tc.action)
				require.NoError(t, err)
				requireProtoEqual(t, tc.result, result)
			})
		}
	})
}

func TestMakeDestroyPolicy(t *testing.T) {
	t.Run("error", func(t *testing.T) {
		t.Parallel()

		_, err := makeDestroyPolicy(&v1proto.Spec_Action{
			Action: &v1proto.Spec_Action_Command{
				Command: &v1proto.Spec_Command{
					Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
				},
			},
		})
		require.Error(t, err)
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name   string
			action *v1proto.Spec_Action
			result *podagent.TDestroyPolicy
		}{
			{
				name:   "nil action",
				action: nil,
				result: nil,
			},
			{
				name:   "empty action",
				action: &v1proto.Spec_Action{},
				result: nil,
			},
			{
				name: "command",
				action: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Command{},
				},
				result: &podagent.TDestroyPolicy{
					Backend: &podagent.TDestroyPolicy_Container{},
				},
			},
			{
				name: "http",
				action: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Http{
						Http: &podagent.THttpGet{Port: 42},
					},
				},
				result: &podagent.TDestroyPolicy{
					Backend: &podagent.TDestroyPolicy_HttpGet{
						HttpGet: &podagent.THttpGet{Port: 42},
					},
				},
			},
			{
				name: "signal",
				action: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Signal{
						Signal: &podagent.TUnixSignal{},
					},
				},
				result: nil,
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				result, err := makeDestroyPolicy(tc.action)
				require.NoError(t, err)
				requireProtoEqual(t, tc.result, result)
			})
		}
	})
}

func TestMakeChecks(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name      string
			readiness *v1proto.Spec_Probe
			liveness  *v1proto.Spec_Probe
		}{
			{
				name: "invalid readiness",
				readiness: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Command{
						Command: &v1proto.Spec_Command{
							Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
						},
					},
				},
				liveness: nil,
			},
			{
				name:      "invalid liveness",
				readiness: nil,
				liveness: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Command{
						Command: &v1proto.Spec_Command{
							Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
						},
					},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				_, _, err := makeChecks(tc.readiness, tc.liveness)
				require.Error(t, err)
			})
		}
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		readinessResult, livenessResult, err := makeChecks(nil, nil)
		require.NoError(t, err)
		requireProtoEqual(t, DefaultReadinessCheck, readinessResult)
		requireProtoEqual(t, DefaultLivenessCheck, livenessResult)
	})
}

func TestMakePolicies(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name    string
			stop    *v1proto.Spec_Action
			destroy *v1proto.Spec_Action
		}{
			{
				name: "invalid stop",
				stop: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Command{
						Command: &v1proto.Spec_Command{
							Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
						},
					},
				},
				destroy: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Http{
						Http: &podagent.THttpGet{Port: 42},
					},
				},
			},
			{
				name: "invalid destroy",
				stop: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Http{
						Http: &podagent.THttpGet{Port: 42},
					},
				},
				destroy: &v1proto.Spec_Action{
					Action: &v1proto.Spec_Action_Command{
						Command: &v1proto.Spec_Command{
							Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
						},
					},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				_, _, err := makePolicies(tc.stop, tc.destroy)
				require.Error(t, err)
			})
		}
	})

	t.Run("ok", func(t *testing.T) {
		t.Parallel()

		stopResult, destroyResult, err := makePolicies(
			&v1proto.Spec_Action{
				Action: &v1proto.Spec_Action_Http{
					Http: &podagent.THttpGet{Port: 42},
				},
			},
			&v1proto.Spec_Action{
				Action: &v1proto.Spec_Action_Http{
					Http: &podagent.THttpGet{Port: 43},
				},
			},
		)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			&podagent.TStopPolicy{
				Backend: &podagent.TStopPolicy_HttpGet{
					HttpGet: &podagent.THttpGet{Port: 42},
				},
			},
			stopResult,
		)
		requireProtoEqual(
			t,
			&podagent.TDestroyPolicy{
				Backend: &podagent.TDestroyPolicy_HttpGet{
					HttpGet: &podagent.THttpGet{Port: 43},
				},
			},
			destroyResult,
		)
	})
}

func TestMakeWorkload(t *testing.T) {
	t.Parallel()

	t.Run("error", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name     string
			workload *v1proto.Spec_Workload
		}{
			{
				name: "invalid init",
				workload: &v1proto.Spec_Workload{
					Init: []*v1proto.Spec_Command{
						{Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"}},
					},
				},
			},
			{
				name: "invalid command",
				workload: &v1proto.Spec_Workload{
					Command: &v1proto.Spec_Command{
						Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
					},
				},
			},
			{
				name: "invalid check",
				workload: &v1proto.Spec_Workload{
					Liveness: &v1proto.Spec_Probe{
						Probe: &v1proto.Spec_Probe_Command{
							Command: &v1proto.Spec_Command{
								Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
							},
						},
					},
				},
			},
			{
				name: "invalid policy",
				workload: &v1proto.Spec_Workload{
					Actions: &v1proto.Spec_Actions{
						Stop: &v1proto.Spec_Action{
							Action: &v1proto.Spec_Action_Command{
								Command: &v1proto.Spec_Command{
									Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
								},
							},
						},
					},
				},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				_, _, err := makeWorkload(tc.workload, "test", "box")
				require.Error(t, err)
			})
		}
	})

	t.Run("unfilled", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			name     string
			workload *v1proto.Spec_Workload
		}{
			{
				name:     "nil workload",
				workload: nil,
			},
			{
				name:     "empty workload",
				workload: &v1proto.Spec_Workload{},
			},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(tc.name, func(t *testing.T) {
				t.Parallel()
				workload, _, err := makeWorkload(tc.workload, "test", "box")
				require.NoError(t, err)

				requireProtoEqual(
					t,
					&podagent.TWorkload{
						Id:             "test",
						BoxRef:         "box",
						LivenessCheck:  DefaultLivenessCheck,
						ReadinessCheck: DefaultReadinessCheck,
						TransmitLogs:   true,
					},
					workload,
				)
			})
		}
	})

	t.Run("command", func(t *testing.T) {
		t.Parallel()

		workload, _, err := makeWorkload(
			&v1proto.Spec_Workload{
				Command: &v1proto.Spec_Command{Cmd: "start"},
			},
			"test", "box",
		)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			&podagent.TUtilityContainer{
				CommandLine:      "start",
				ComputeResources: &podagent.TComputeResources{},
			},
			workload.Start,
		)
	})

	t.Run("readiness", func(t *testing.T) {
		t.Parallel()

		workload, _, err := makeWorkload(
			&v1proto.Spec_Workload{
				Readiness: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Command{
						Command: &v1proto.Spec_Command{Cmd: "readiness"},
					},
				},
			},
			"test", "box",
		)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			&podagent.TReadinessCheck{
				Backend: &podagent.TReadinessCheck_Container{
					Container: &podagent.TUtilityContainer{
						CommandLine:      "readiness",
						ComputeResources: &podagent.TComputeResources{},
					},
				},
			},
			workload.ReadinessCheck,
		)
	})

	t.Run("liveness", func(t *testing.T) {
		t.Parallel()

		workload, _, err := makeWorkload(
			&v1proto.Spec_Workload{
				Liveness: &v1proto.Spec_Probe{
					Probe: &v1proto.Spec_Probe_Command{
						Command: &v1proto.Spec_Command{Cmd: "liveness"},
					},
				},
			},
			"test", "box",
		)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			&podagent.TLivenessCheck{
				Backend: &podagent.TLivenessCheck_Container{
					Container: &podagent.TUtilityContainer{
						CommandLine:      "liveness",
						ComputeResources: &podagent.TComputeResources{},
					},
				},
			},
			workload.LivenessCheck,
		)
	})

	t.Run("stop", func(t *testing.T) {
		t.Parallel()

		workload, _, err := makeWorkload(
			&v1proto.Spec_Workload{
				Actions: &v1proto.Spec_Actions{
					Stop: &v1proto.Spec_Action{
						Action: &v1proto.Spec_Action_Command{
							Command: &v1proto.Spec_Command{Cmd: "stop"},
						},
					},
				},
			},
			"test", "box",
		)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			&podagent.TStopPolicy{
				Backend: &podagent.TStopPolicy_Container{
					Container: &podagent.TUtilityContainer{
						CommandLine:      "stop",
						ComputeResources: &podagent.TComputeResources{},
					},
				},
			},
			workload.StopPolicy,
		)
	})

	t.Run("destroy", func(t *testing.T) {
		t.Parallel()

		workload, _, err := makeWorkload(
			&v1proto.Spec_Workload{
				Actions: &v1proto.Spec_Actions{
					Destroy: &v1proto.Spec_Action{
						Action: &v1proto.Spec_Action_Command{
							Command: &v1proto.Spec_Command{Cmd: "destroy"},
						},
					},
				},
			},
			"test", "box",
		)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			&podagent.TDestroyPolicy{
				Backend: &podagent.TDestroyPolicy_Container{
					Container: &podagent.TUtilityContainer{
						CommandLine:      "destroy",
						ComputeResources: &podagent.TComputeResources{},
					},
				},
			},
			workload.DestroyPolicy,
		)
	})

	t.Run("logs", func(t *testing.T) {
		t.Parallel()

		testCases := []struct {
			mode         v1proto.Spec_LogsTransmission_Mode
			transmitLogs bool
		}{
			{v1proto.Spec_LogsTransmission_LOGS_DISABLED, false},
			{v1proto.Spec_LogsTransmission_LOGS_ENABLED, true},
			{v1proto.Spec_LogsTransmission_LOGS_UNKNOWN, true},
		}

		for _, tc := range testCases {
			tc := tc
			t.Run(fmt.Sprintf("%T", tc.mode), func(t *testing.T) {
				t.Parallel()
				workload, _, err := makeWorkload(
					&v1proto.Spec_Workload{
						Logs: &v1proto.Spec_LogsTransmission{
							Enabled: tc.mode,
						},
					},
					"test", "box",
				)
				require.NoError(t, err)
				require.Equal(t, workload.TransmitLogs, tc.transmitLogs)
			})
		}
	})

	t.Run("init", func(t *testing.T) {
		t.Parallel()

		workload, _, err := makeWorkload(
			&v1proto.Spec_Workload{
				Init: []*v1proto.Spec_Command{{Cmd: "init"}},
			},
			"test", "box",
		)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			[]*podagent.TUtilityContainer{
				{
					CommandLine:      "init",
					ComputeResources: &podagent.TComputeResources{},
				},
			},
			workload.Init,
		)
	})

	t.Run("env", func(t *testing.T) {
		t.Parallel()

		workload, refs, err := makeWorkload(
			&v1proto.Spec_Workload{
				Env: map[string]string{
					"literal": "value",
					"secret":  "${secid:secver:seckey}",
				},
			},
			"test", "box",
		)
		require.NoError(t, err)

		requireProtoEqual(
			t,
			[]*podagent.TEnvVar{
				{
					Name: "literal",
					Value: &podagent.TEnvVarValue{
						Value: &podagent.TEnvVarValue_LiteralEnv{
							LiteralEnv: &podagent.LiteralEnvSelector{Value: "value"},
						},
					},
				},
				{
					Name: "secret",
					Value: &podagent.TEnvVarValue{
						Value: &podagent.TEnvVarValue_SecretEnv{
							SecretEnv: &podagent.SecretSelector{
								Alias: "secid:secver",
								Id:    "seckey",
							},
						},
					},
				},
			},
			workload.Env,
		)

		requireProtoEqual(
			t,
			map[string]*ypapi.TSecretRef{
				"secid:secver": {
					SecretId:      ptr.String("secid"),
					SecretVersion: ptr.String("secver"),
				},
			},
			refs,
		)
	})
}

func TestMakeWorkloads(t *testing.T) {
	t.Run("no workloads", func(t *testing.T) {
		t.Parallel()

		t.Run("error", func(t *testing.T) {
			testCases := []struct {
				name string
				spec *v1proto.Spec
			}{
				{
					name: "invalid check",
					spec: &v1proto.Spec{
						Readiness: &v1proto.Spec_Probe{
							Probe: &v1proto.Spec_Probe_Command{
								Command: &v1proto.Spec_Command{
									Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
								},
							},
						},
					},
				},
				{
					name: "invalid policy",
					spec: &v1proto.Spec{
						Lifecycle: &v1proto.Spec_Actions{
							Stop: &v1proto.Spec_Action{
								Action: &v1proto.Spec_Action_Command{
									Command: &v1proto.Spec_Command{
										Resources: &v1proto.Spec_ComputeResources{Vcpu: "invalid"},
									},
								},
							},
						},
					},
				},
			}

			for _, tc := range testCases {
				tc := tc
				t.Run(tc.name, func(t *testing.T) {
					t.Parallel()

					_, _, err := makeWorkloads(&deployinfrav1.Runtime{
						Spec:       tc.spec,
						ObjectMeta: v1.ObjectMeta{Name: "test"},
					})
					require.Error(t, err)
				})
			}
		})

		t.Run("ok", func(t *testing.T) {
			t.Parallel()

			wl, _, err := makeWorkloads(
				&deployinfrav1.Runtime{
					ObjectMeta: v1.ObjectMeta{Name: "test"},
					Spec: &v1proto.Spec{
						Readiness: &v1proto.Spec_Probe{
							Probe: &v1proto.Spec_Probe_Command{
								Command: &v1proto.Spec_Command{Cmd: "readiness"},
							},
						},
						Liveness: &v1proto.Spec_Probe{
							Probe: &v1proto.Spec_Probe_Command{
								Command: &v1proto.Spec_Command{Cmd: "liveness"},
							},
						},
						Lifecycle: &v1proto.Spec_Actions{
							Stop: &v1proto.Spec_Action{
								Action: &v1proto.Spec_Action_Command{
									Command: &v1proto.Spec_Command{Cmd: "stop"},
								},
							},
							Destroy: &v1proto.Spec_Action{
								Action: &v1proto.Spec_Action_Command{
									Command: &v1proto.Spec_Command{Cmd: "destroy"},
								},
							},
						},
						Command: "cmd",
						Logs:    &v1proto.Spec_LogsTransmission{Enabled: v1proto.Spec_LogsTransmission_LOGS_DISABLED},
					},
				},
			)
			require.NoError(t, err)

			requireProtoEqual(
				t,
				[]*podagent.TWorkload{
					{
						Id:     "test",
						BoxRef: "test",
						Env:    []*podagent.TEnvVar{},
						Start:  &podagent.TUtilityContainer{CommandLine: "cmd"},
						ReadinessCheck: &podagent.TReadinessCheck{
							Backend: &podagent.TReadinessCheck_Container{
								Container: &podagent.TUtilityContainer{
									ComputeResources: &podagent.TComputeResources{},
									CommandLine:      "readiness",
								},
							},
						},
						LivenessCheck: &podagent.TLivenessCheck{
							Backend: &podagent.TLivenessCheck_Container{
								Container: &podagent.TUtilityContainer{
									ComputeResources: &podagent.TComputeResources{},
									CommandLine:      "liveness",
								},
							},
						},
						StopPolicy: &podagent.TStopPolicy{
							Backend: &podagent.TStopPolicy_Container{
								Container: &podagent.TUtilityContainer{
									ComputeResources: &podagent.TComputeResources{},
									CommandLine:      "stop",
								},
							},
						},
						DestroyPolicy: &podagent.TDestroyPolicy{
							Backend: &podagent.TDestroyPolicy_Container{
								Container: &podagent.TUtilityContainer{
									ComputeResources: &podagent.TComputeResources{},
									CommandLine:      "destroy",
								},
							},
						},
						TransmitLogs: false,
					},
				},
				wl,
			)
		})
	})

	t.Run("command with workloads", func(t *testing.T) {
		t.Parallel()

		_, _, err := makeWorkloads(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Command:   "cmd",
				Workloads: map[string]*v1proto.Spec_Workload{"workload": {}},
			},
			ObjectMeta: v1.ObjectMeta{Name: "test"},
		})
		require.Error(t, err)
	})

	t.Run("workloads", func(t *testing.T) {
		t.Parallel()

		wl, _, err := makeWorkloads(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Workloads: map[string]*v1proto.Spec_Workload{"w1": {}, "w2": {}},
			},
			ObjectMeta: v1.ObjectMeta{Name: "test"},
		})
		require.NoError(t, err)
		require.Equal(t, 2, len(wl), "incorrect number of workloads")

		assert.Equal(t, wl[0].Id, "w1")
		assert.Equal(t, wl[1].Id, "w2")
	})

	t.Run("secret refs", func(t *testing.T) {
		t.Parallel()

		_, refs, err := makeWorkloads(&deployinfrav1.Runtime{
			Spec: &v1proto.Spec{
				Workloads: map[string]*v1proto.Spec_Workload{
					"w1": {
						Env: map[string]string{
							"env1": "${secid1:secver1:seckey1}",
						},
					},
					"w2": {
						Env: map[string]string{
							"env2": "${secid2:secver2:seckey2}",
						},
					},
				},
			},
			ObjectMeta: v1.ObjectMeta{Name: "test"},
		})
		require.NoError(t, err)

		requireProtoEqual(
			t,
			map[string]*ypapi.TSecretRef{
				"secid1:secver1": {
					SecretId:      ptr.String("secid1"),
					SecretVersion: ptr.String("secver1"),
				},
				"secid2:secver2": {
					SecretId:      ptr.String("secid2"),
					SecretVersion: ptr.String("secver2"),
				},
			},
			refs,
		)
	})
}

func TestLogsEnabled(t *testing.T) {
	t.Parallel()

	testCases := []struct {
		mode     v1proto.Spec_LogsTransmission_Mode
		expected bool
	}{
		{v1proto.Spec_LogsTransmission_LOGS_ENABLED, true},
		{v1proto.Spec_LogsTransmission_LOGS_DISABLED, false},
		{v1proto.Spec_LogsTransmission_LOGS_UNKNOWN, true},
	}

	for _, tc := range testCases {
		t.Run(v1proto.Spec_LogsTransmission_Mode_name[int32(tc.mode)], func(t *testing.T) {
			require.Equal(
				t,
				logsEnabled(&v1proto.Spec_LogsTransmission{Enabled: tc.mode}),
				tc.expected,
			)
		})
	}
}
