package main

import (
	"os"

	"github.com/spf13/pflag"
	"k8s.io/client-go/kubernetes/scheme"
	// Import all Kubernetes client auth plugins (e.g. Azure, GCP, OIDC, etc.)
	// to ensure that exec-entrypoint and run can make use of them.
	_ "k8s.io/client-go/plugin/pkg/client/auth"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/healthz"

	deployinfrav1 "a.yandex-team.ru/infra/infractl/controllers/runtime/api/v1"
	"a.yandex-team.ru/infra/infractl/controllers/runtime/controllers"
	"a.yandex-team.ru/infra/infractl/internal/log"
	_ "a.yandex-team.ru/infra/infractl/internal/scheme"
)

var (
	setupLog = ctrl.Log.WithName("setup")
)

func main() {
	configFile := pflag.String("config", "",
		"The controller will load its initial configuration from this file. "+
			"Omit this flag to use the default configuration values. "+
			"Command-line flags override configuration from this file.")

	logopts := log.Bind(pflag.CommandLine)

	pflag.Parse()

	ctrl.SetLogger(logopts.ConfigureLogger())

	var err error
	ctrlConfig := deployinfrav1.ControllerConfig{}
	options := ctrl.Options{Scheme: scheme.Scheme}
	if *configFile != "" {
		options, err = options.AndFrom(ctrl.ConfigFile().AtPath(*configFile).OfKind(&ctrlConfig))
		if err != nil {
			setupLog.Error(err, "unable to load config file")
			os.Exit(1)
		}
	}

	mgr, err := ctrl.NewManager(ctrl.GetConfigOrDie(), options)

	if err != nil {
		setupLog.Error(err, "unable to start manager", "config", ctrl.GetConfigOrDie())
		os.Exit(1)
	}

	reconciler := &controllers.RuntimeReconciler{
		Client:  mgr.GetClient(),
		Scheme:  mgr.GetScheme(),
		Stats:   controllers.MakeStats(),
		Options: ctrlConfig.Options,
	}

	if err = mgr.AddMetricsExtraHandler(ctrlConfig.Unistat.Path, reconciler.Stats); err != nil {
		setupLog.Error(err, "unable to setup unistat", "controller", "Runtime")
		os.Exit(1)
	}

	if err = reconciler.SetupWithManager(mgr); err != nil {
		setupLog.Error(err, "unable to create controller", "controller", "Runtime")
		os.Exit(1)
	}

	if err := mgr.AddHealthzCheck("healthz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up health check")
		os.Exit(1)
	}
	if err := mgr.AddReadyzCheck("readyz", healthz.Ping); err != nil {
		setupLog.Error(err, "unable to set up ready check")
		os.Exit(1)
	}

	setupLog.Info("starting manager")
	if err := mgr.Start(ctrl.SetupSignalHandler()); err != nil {
		setupLog.Error(err, "problem running manager")
		os.Exit(1)
	}
}
