from textwrap import dedent, indent


def write_gvk(crd):
    print(dedent(
        # FIXME (torkve) currently version is always v1
        f"""\
        var (
            // GroupVersion is group version used to register these objects
            GroupVersion = schema.GroupVersion{{Group: "{crd.api_group}", Version: "v1"}}

            // SchemeBuilder is used to add go types to the GroupVersionKind scheme
            SchemeBuilder = &scheme.Builder{{GroupVersion: GroupVersion}}

            // AddToScheme adds the types in this group-version to the given scheme.
            AddToScheme = SchemeBuilder.AddToScheme
        )
        """
    ))


def write_struct(name, has_spec, has_status):
    print(dedent(
        f"""\
        type {name} struct {{
            metav1.TypeMeta   `json:",inline"`
            metav1.ObjectMeta `json:"metadata,omitempty"`
        """
    ))
    if has_spec:
        print('    Spec   *proto_v1.Spec   `json:"spec,omitempty"`')
    if has_status:
        print('    Status *proto_v1.Status `json:"status,omitempty"`')
    print("}\n")

    print(dedent(
        f"""\
            var {name}TypeMeta = metav1.TypeMeta {{
                Kind:       "{name}",
                APIVersion: GroupVersion.String(),
            }}
        """
    ))


def write_constructor(name):
    print(dedent(
        f"""\
        func New{name}() *{name} {{
            return &{name} {{
                TypeMeta: {name}TypeMeta,
            }}
        }}
        """
    ))


def write_getters(name, has_spec, has_status):
    if has_spec:
        print(dedent(
            f"""\
            func (o *{name}) GetSpec() *proto_v1.Spec {{
                if o == nil {{
                    return nil
                }}
                return o.Spec
            }}
            """
        ))

    if has_status:
        print(dedent(
            f"""\
            func (o *{name}) GetStatus() *proto_v1.Status {{
                if o == nil {{
                    return nil
                }}
                return o.Status
            }}
            """
        ))


def write_templatable_interface(name, has_status):
    if has_status:
        print(dedent(
            f"""\
            func (o *{name}) ClearStatus() {{
                o.Status = nil
            }}
            """
        ))


def write_unmarshal(name, has_spec, has_status):
    print(dedent(
        f"""\
        func (o *{name}) UnmarshalJSON(bytes []byte) error {{
            var data map[string]*json.RawMessage
            var err error
            if err = json.Unmarshal(bytes, &data); err != nil {{
                return err
            }}
        """
    ))

    if has_spec:
        print(indent(dedent(
            """\
            o.Spec = &proto_v1.Spec{}
            specValue := data["spec"]
            var spec []byte
            if specValue != nil {
                spec, err = specValue.MarshalJSON()
                if err != nil {
                    return err
                }
            }
            """
        ), '    '))

    if has_status:
        print(indent(dedent(
            """\
            o.Status = &proto_v1.Status{}
            statusValue := data["status"]
            var status []byte
            if statusValue != nil {
                status, err = data["status"].MarshalJSON()
                if err != nil {
                    return err
                }
            }
            """
        ), '    '))

    print(indent(dedent(
        """\

        if err := json.Unmarshal(*data["metadata"], &o.ObjectMeta); err != nil {
            return err
        }

        delete(data, "spec")
        delete(data, "status")
        delete(data, "metadata")

        bytes, err = json.Marshal(data)
        if err != nil {
            return err
        }
        if err := json.Unmarshal(bytes, &o.TypeMeta); err != nil {
            return err
        }

        options := protojson.UnmarshalOptions{
            AllowPartial:   true,
            DiscardUnknown: true,
        }
        """
    ), '    '))
    if has_spec:
        print(indent(dedent(
            """\

            if specValue != nil {
                if err = options.Unmarshal(spec, o.Spec); err != nil {
                    return err
                }
            }
            """
        ), '    '))
    if has_status:
        print(indent(dedent(
            """\
            if statusValue != nil {
                if err = options.Unmarshal(status, o.Status); err != nil {
                    return err
                }
            }
            """
        ), '    '))

    print(dedent(
        """\
            if o.Annotations == nil {
                o.Annotations = make(map[string]string)
            }

            return nil
        }

        """
    ))


def write_marshal(name, has_spec, has_status):
    print(dedent(
        f"""\
        func (o *{name}) MarshalJSON() ([]byte, error) {{
            options := protojson.MarshalOptions{{
                Multiline:       false,
                AllowPartial:    true,
                UseProtoNames:   true,
                UseEnumNumbers:  false,
                EmitUnpopulated: false,
            }}
        """
    ))
    if has_spec:
        print(indent(dedent(
            """\
            var rawSpec *json.RawMessage
            if o.Spec != nil {
                rawSpec = &json.RawMessage{}
                spec, err := options.Marshal(o.Spec)
                if err != nil {
                    return nil, err
                }
                if err = json.Unmarshal(spec, &rawSpec); err != nil {
                    return nil, err
                }
            }
            """
        ), '    '))
    if has_status:
        print(indent(dedent(
            """\
            var rawStatus *json.RawMessage
            if o.Status != nil {
                rawStatus = &json.RawMessage{}
                status, err := options.Marshal(o.Status)
                if err != nil {
                    return nil, err
                }
                if err = json.Unmarshal(status, &rawStatus); err != nil {
                    return nil, err
                }
            }
            """
        ), '    '))

    print(indent(dedent(
        """\
        return json.Marshal(&struct {
            metav1.TypeMeta   `json:",inline"`
            metav1.ObjectMeta `json:"metadata,omitempty"`
        """
    ), '    '))
    if has_spec:
        print(indent('Spec              *json.RawMessage `json:"spec,omitempty"`', '        '))
    if has_status:
        print(indent('Status            *json.RawMessage `json:"status,omitempty"`', '        '))
    print(dedent(
        """\
            }{
                TypeMeta:   o.TypeMeta,
                ObjectMeta: o.ObjectMeta,
                Spec:       rawSpec,
                Status:     rawStatus,
            })
        }
        """
    ))


def write_type_list(name):
    print(dedent(
        f"""\
        // {name}List contains a list of {name}
        type {name}List struct {{
            metav1.TypeMeta `json:",inline"`
            metav1.ListMeta `json:"metadata,omitempty"`
            Items           []{name} `json:"items"`
        }}
        """
    ))


def write_init(name):
    print(dedent(
        f"""\
        func init() {{
            SchemeBuilder.Register(&{name}{{}}, &{name}List{{}})
        }}
        """
    ))
