import sys
import importlib
from textwrap import dedent

from library.go.k8s.protoc_gen_crd.proto.crd_pb2 import k8s_crd

from infra.infractl.crd_to_go import generators, deepcopy_generators


def main():
    if len(sys.argv) != 3:
        raise RuntimeError(f"This generator must be run as {sys.argv[0]} {{PROTO_MODULE}} {{GO_PACKAGE}}")

    proto_module = sys.argv[1]
    go_package = sys.argv[2].rsplit('/', 1)[-1]  # user may pass 'v1' or 'a.yandex.team.ru/a/b/v1'

    module = importlib.import_module(proto_module)
    if not hasattr(module, 'DESCRIPTOR'):
        raise RuntimeError(f"Module {proto_module!r} is not a valid protobuf module")

    go_proto_module = module.DESCRIPTOR.GetOptions().go_package

    print(dedent(
        f"""\
        //go:build !ignore_autogenerated

        // Code is generated by infra/infractl/crd_to_go.
        // Autogenerated file, do not edit.

        package {go_package}

        import (
            "encoding/json"

            "google.golang.org/protobuf/encoding/protojson"
            metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
            runtime "k8s.io/apimachinery/pkg/runtime"
            "k8s.io/apimachinery/pkg/runtime/schema"
            "sigs.k8s.io/controller-runtime/pkg/scheme"

            proto_v1 "{go_proto_module}"
        )
        """
    ))

    # TODO (torkve) check that it's exactly one api group in the file. Currently we expect exactly one CRD at all.
    for message_name, message_descriptor in module.DESCRIPTOR.message_types_by_name.iteritems():
        extensions = message_descriptor.GetOptions().Extensions
        if k8s_crd not in extensions:
            continue

        crd = extensions[k8s_crd]

        has_spec = 'spec' in message_descriptor.fields_by_name
        has_status = 'status' in message_descriptor.fields_by_name

        generators.write_gvk(crd)
        generators.write_struct(message_name, has_spec, has_status)
        generators.write_constructor(message_name)
        generators.write_getters(message_name, has_spec, has_status)
        generators.write_templatable_interface(message_name, has_status)
        generators.write_unmarshal(message_name, has_spec, has_status)
        generators.write_marshal(message_name, has_spec, has_status)
        generators.write_type_list(message_name)

        deepcopy_generators.write_deep_copy_into(message_name, has_spec, has_status)
        deepcopy_generators.write_deep_copy(message_name)
        deepcopy_generators.write_deep_copy_object(message_name)
        deepcopy_generators.write_list_deep_copy_into(message_name)
        deepcopy_generators.write_list_deep_copy(message_name)
        deepcopy_generators.write_list_deep_copy_object(message_name)

        generators.write_init(message_name)
