package client

import (
	"context"
	"fmt"

	crclient "sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yt/go/yterrors"
	"a.yandex-team.ru/yt/yt/orm/go/orm/ormerrors"
)

const (
	infractlAnnotationKey           = "/annotations/infractl"
	infractlAnnotationGenerationKey = "/annotations/infractl/generation"
)

type DeployClient struct {
	ypClient     *yp.Client
	userYPClient *yp.Client
	specWriter   crclient.Writer
}

func NewDeployClient(ypClient, userYPClient *yp.Client, specWriter crclient.Writer) *DeployClient {
	return &DeployClient{
		ypClient:     ypClient,
		userYPClient: userYPClient,
		specWriter:   specWriter,
	}
}

func (c *DeployClient) Fetch(
	ctx context.Context,
	timestamp uint64,
	kObj interfaces.KubernetesObject,
) (interfaces.DeployObject, error) {
	req := yp.GetObjectRequest{
		ObjectType:      kObj.GetObjectType(),
		ObjectID:        kObj.GetName(),
		Format:          yp.PayloadFormatProto,
		Selectors:       []string{"/meta", "/spec", "/status", infractlAnnotationKey},
		FetchTimestamps: true,
		FetchRootObject: true,
	}
	if timestamp != 0 {
		req.Timestamp = timestamp
	}
	response, err := c.ypClient.GetObject(ctx, req)
	if err != nil {
		if yterrors.ContainsErrorCode(err, ormerrors.CodeNoSuchObject) {
			return nil, nil
		}
		return nil, fmt.Errorf("failed to fetch %s from YP: %w", kObj.GetReadableObjectType(), err)
	}
	dObj, err := kObj.FillDeployObject(response)
	if err != nil {
		return nil, fmt.Errorf("failed to parse YP response: %w", err)
	}
	return dObj, nil
}

func (c *DeployClient) Create(
	ctx context.Context,
	kObj interfaces.KubernetesObject,
	dObj interfaces.DeployObject,
) (uint64, error) {
	var createRsp *yp.CreateObjectResponse

	ypObj := dObj.GetYPObject()
	// Start YP transaction to make sure that if we fail to store fqid in k8s object then we do not create YP object.
	trID, _, err := c.userYPClient.StartTransaction(ctx)
	if err != nil {
		return 0, fmt.Errorf("failed to start transaction in YP: %w", err)
	}
	createReq := yp.CreateObjectRequest{
		ObjectType:    kObj.GetObjectType(),
		Object:        ypObj,
		TransactionID: trID,
	}
	createRsp, err = c.userYPClient.CreateObject(ctx, createReq)
	if err != nil {
		return 0, fmt.Errorf("failed to create object in YP: %w", err)
	}
	kObj.SetFqid(createRsp.Fqid)
	if err = c.specWriter.Update(ctx, kObj); err != nil {
		return 0, fmt.Errorf("failed to save fqid in k8s: %w", err)
	}
	// If we fail on commit transaction here then we get non-consistent state when fqid is stored in k8s
	// but YP object does not exist. On the next iterations we will try to create object once again
	// and will forcefully overwrite fqid in k8s
	if _, err = c.userYPClient.CommitTransaction(ctx, trID); err != nil {
		return 0, fmt.Errorf("failed to commit transaction %s in YP: %w", trID, err)
	}
	return createRsp.CommitTimestamp, nil
}

func (c *DeployClient) Update(
	ctx context.Context,
	kObj interfaces.KubernetesObject,
	dObj interfaces.DeployObject,
	curYPSpecTimestamp uint64,
) (uint64, error) {
	// Now we do the following:
	// 1. Compare k8s.metadata.generation and yp.annotations.infractl.generation. If they differ:
	//    1.a update YP object spec
	//    1.b Put k8s.metadata.generation into yp.annotations.infractl.generation
	// 2. Read object from YP (from timestamp = update_response.commit_timestamp)
	// 3. Set:
	//    3.a k8s.status.appliedGeneration = metadata.generation
	//    3.b k8s.status.stage_status.revisions = yp.stage.status.revisions
	//
	// We do steps (2) and (3) even if we didn't steps (1.a) and (1.b).
	//
	// But this behavior is racy: if we die between steps 2 and 3 and user update spec in YP
	// then we will set status.stage_revisions equal to revisions from YP and
	// it will look like we committed our spec to YP and everything is fine but indeed
	// our spec was overwritten by user.
	//
	// To avoid it may be we should do the following:
	// 1. Start transaction in YP
	// 2. Update spec and yp.annotations.infractl.generation in YP
	// 3. Read stage revisions from this transaction (not supported in YP)
	// 4. Set k8s.status.stage_status.revisions = yp.stage.status.revisions
	// 5. Commit transaction in YP
	// 6. Read object from YP (from timestamp = update_response.commit_timestamp)
	// 7. Set k8s.status.appliedGeneration = metadata.generation

	// Update generation stored in YP annotations to make sure we applied spec from k8s to YP.
	// We will use this generation in SpecNewer function on the next iteration.
	ypSpec := dObj.GetYPSpec()

	ret, err := c.userYPClient.UpdateObject(ctx, yp.UpdateObjectRequest{
		ObjectType: kObj.GetObjectType(),
		ObjectID:   kObj.GetName(),
		SetUpdates: []yp.SetObjectUpdate{
			{
				Path:   "/spec",
				Object: ypSpec,
			},
			{
				Path:   infractlAnnotationGenerationKey,
				Object: kObj.GetGeneration(),
			},
		},
		AttributeTimestampPrerequisites: []yp.PrerequisiteObjectUpdate{{
			Path:      "/spec",
			Timestamp: curYPSpecTimestamp,
		}},
	})
	if err != nil {
		return 0, err
	}
	return ret.CommitTimestamp, nil
}
