package validation

import (
	"context"
	"fmt"
	"reflect"
	"strings"

	"github.com/go-logr/logr"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/deploy/interfaces"
)

func isNil(i interface{}) bool {
	return i == nil || reflect.ValueOf(i).IsNil()
}

func ValidateFqidMatchesYP(
	ctx context.Context,
	deployClient *dclient.DeployClient,
	log logr.Logger,
	kObj interfaces.KubernetesObject,
) (bool, string, error) {
	newFqid := kObj.GetFqid()

	oType := strings.ToLower(kObj.GetReadableObjectType())
	dObj, err := deployClient.Fetch(ctx, 0, kObj)
	if err != nil {
		log.Error(err, "failed to fetch YP object")
		return false, "", fmt.Errorf("failed to fetch YP %s %q: %w", oType, kObj.GetName(), err)
	}

	var curFqid string
	if dObj == nil {
		// It may happen that when creating YP object we failed after we stored fqid in k8s object but before
		// committing transaction in YP. In that case YP object doesn't exist but fqid is stored in k8s object.
		// We don't check fqid because it will be overwritten in k8s after creating object in YP on the next
		// controller iteration.
		log.Info("object in YP does not exist, skip checking if fqid matches to YP")
		return true, "", nil
	} else {
		curFqid = dObj.GetFqid()
	}

	if curFqid != newFqid {
		var msg string
		if newFqid == "" {
			msg = fmt.Sprintf(
				"Fqid cannot be empty because %s %q already exists in YP with fqid %q",
				oType, dObj.GetID(), curFqid,
			)
		} else {
			msg = fmt.Sprintf("Fqid %q does not match fqid of %s in YP %q", newFqid, oType, curFqid)
		}
		return false, msg, nil
	}
	return true, "", nil
}

func ValidateAccountIDMatchesYP(
	ctx context.Context,
	log logr.Logger,
	deployClient *dclient.DeployClient,
	abcClient *abc.Client,
	abcSlug string,
	kObj interfaces.KubernetesObject,
) (bool, string, error) {
	oType := strings.ToLower(kObj.GetReadableObjectType())
	dObj, err := deployClient.Fetch(ctx, 0, kObj)
	if err != nil {
		log.Error(err, "failed to fetch YP object")
		return false, "", fmt.Errorf("failed to fetch YP %s %q: %w", oType, kObj.GetName(), err)
	}

	accIDInt, err := abcClient.GetServiceIDBySlug(abcSlug)
	if err != nil {
		return false, "", fmt.Errorf("failed to get ABC service ID by slug %q: %w", abcSlug, err)
	}
	accID := fmt.Sprintf("abc:service:%d", accIDInt)

	if dObj != nil {
		// Check if namespace ABC service is equal to YP obj ABC account, because deploy controller will overwrite
		// ABC account in YP object if it differs from K8S namespace ABC account.
		ypAccID := dObj.GetAccountID()
		// account_id is required only for YP projects. For other YP objects it may be empty
		// (and inherited from project). So do not validate account_id if it is empty in YP.
		if ypAccID != "" && ypAccID != accID {
			return false, fmt.Sprintf("Namespace ABC account %q is not equal to YP %s ABC account %q", accID, oType, ypAccID), nil
		}
	}
	return true, "", nil
}

func ValidateDeployObject(
	ctx context.Context,
	log logr.Logger,
	deployClient *dclient.DeployClient,
	abcClient *abc.Client,
	abcSlug string,
	curKObj, newKObj interfaces.KubernetesObject,
) (bool, string, error) {

	gotAccID := newKObj.GetAccountID()
	if gotAccID != "" {
		msg := fmt.Sprintf(
			"account ID cannot be set in %s spec, got %q, please clear it, account specified in namespace "+
				"will be used instead", strings.ToLower(newKObj.GetReadableObjectType()), gotAccID)
		return false, msg, nil
	}

	var curFqid string
	if isNil(curKObj) {
		isValid, msg, err := ValidateAccountIDMatchesYP(ctx, log, deployClient, abcClient, abcSlug, newKObj)
		if err != nil || !isValid {
			return isValid, msg, err
		}
		curFqid = ""
	} else {
		curFqid = curKObj.GetFqid()
	}

	if curFqid != newKObj.GetFqid() {
		log.Info("fqid changed", "old", curFqid, "new", newKObj.GetFqid())
		return ValidateFqidMatchesYP(ctx, deployClient, log, newKObj)
	}

	return true, "", nil
}
