package log

import (
	"fmt"
	"os"

	"github.com/go-logr/logr"
	"github.com/go-logr/zapr"
	"github.com/spf13/pflag"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

type Options struct {
	LogLevel  int
	IsConsole bool
}

func Bind(set *pflag.FlagSet) *Options {
	o := &Options{}
	set.IntVar(&o.LogLevel, "log-level", int(log.InfoLevel), "Log level to use")
	set.BoolVarP(&o.IsConsole, "console", "c", false, "Use human-readable log output for debugging")
	return o
}

func (o *Options) ConfigureLogger() logr.Logger {
	return ConfigureLogger(log.Level(o.LogLevel), o.IsConsole)
}

func ConfigureZapLogger(logLevel log.Level, isConsole bool) *zap.Logger {
	var err error
	var log *zap.Logger
	if isConsole {
		cfg := zap.CLIConfig(logLevel)
		log, err = zap.New(cfg)
	} else {
		log, err = zap.NewDeployLogger(logLevel)
	}
	if err != nil {
		fmt.Printf("Cannot configure logger: %v\n", err)
		os.Exit(1)
	}

	return log
}

func ConfigureLogger(logLevel log.Level, isConsole bool) logr.Logger {
	zapLog := ConfigureZapLogger(logLevel, isConsole)
	return zapr.NewLogger(zapLog.L)
}
