package secrets

import (
	"context"
	"crypto/tls"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
)

const (
	DefaultEndpoint = "https://oauth.yandex-team.ru/token"
	DefaultTimeout  = 5 * time.Second
)

type (
	oauthResponse struct {
		Token string `json:"access_token"`
	}

	oauthErrorResponse struct {
		Error string `json:"error_description"`
	}
)

func newRestyClient() *resty.Client {
	r := resty.New()
	certPool, err := certifi.NewCertPool()
	if err == nil {
		r.SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	}

	return r
}

func tryToGetToken(ctx context.Context, httpc *resty.Client, endpoint string, postData url.Values) (string, error) {
	resp, err := httpc.R().
		SetContext(ctx).
		SetHeader("Content-Type", "application/x-www-form-urlencoded").
		SetBody(postData.Encode()).
		Post(endpoint)
	if err != nil {
		return "", fmt.Errorf("failed to call oauth server: %w", err)
	}

	switch resp.StatusCode() {
	case http.StatusOK:
		var result oauthResponse
		if err := json.Unmarshal(resp.Body(), &result); err != nil {
			return "", fmt.Errorf("failed to parse oauth response: %q", string(resp.Body()))
		}
		return result.Token, nil
	case http.StatusBadRequest:
		var result oauthErrorResponse
		if err := json.Unmarshal(resp.Body(), &result); err != nil {
			return "", fmt.Errorf("failed to parse oauth response: %q", string(resp.Body()))
		}
		return "", fmt.Errorf("failed to get OAuth token: %q", result.Error)
	default:
		return "", fmt.Errorf("unexpected response status: %d", resp.StatusCode())
	}
}

type (
	Option func(*options) error

	options struct {
		endpoint string
		timeout  time.Duration
	}
)

// WithEndpoint sets custom endpoint.
func WithEndpoint(endpoint string) Option {
	return func(opts *options) error {
		opts.endpoint = endpoint
		return nil
	}
}

// WithTimeout sets timeout for all of exchange.
//
// Default: 5 sec
func WithTimeout(timeout time.Duration) Option {
	return func(opts *options) error {
		opts.timeout = timeout
		return nil
	}
}

type Provider struct {
	Name              string `yaml:"name"`
	YAVSecretKey      string `yaml:"yav_secret_key"`
	OAuthClientID     string `yaml:"oauth_client_id"`
	OAuthClientSecret string `yaml:"oauth_client_secret"`
	NeedsDelegation   bool   `yaml:"needs_delegation"`
}

func (p *Provider) GetOAuthTokenByCredentials(ctx context.Context, creds *Credentials, opts ...Option) (string, error) {
	o := &options{
		endpoint: DefaultEndpoint,
		timeout:  DefaultTimeout,
	}
	for _, opt := range opts {
		if err := opt(o); err != nil {
			return "", err
		}
	}

	ctx, cancel := context.WithTimeout(ctx, o.timeout)
	defer cancel()

	httpClient := newRestyClient()
	postData := url.Values{
		"client_id":     {p.OAuthClientID},
		"client_secret": {p.OAuthClientSecret},
		"grant_type":    {"password"},
		"username":      {creds.Username},
		"password":      {creds.Password},
	}

	return tryToGetToken(ctx, httpClient, DefaultEndpoint, postData)
}
