package secrets

import (
	"context"
	"fmt"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/types"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/yav"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

type UserSecretReader interface {
	GetYavClient() *httpyav.Client
	GetK8sClient() client.Client
}

type SecretReader interface {
	UserSecretReader
	GetTvmClient() *tvmauth.Client
}

func LoadProviderTokenAsUser(
	reader UserSecretReader,
	ctx context.Context,
	namespace *corev1.Namespace,
	provider string,
) (token string, err error) {
	secretName := MakeK8sSecretName(provider)
	var kSecret corev1.Secret
	typedSecretName := types.NamespacedName{Namespace: namespace.Name, Name: secretName}
	if err = reader.GetK8sClient().Get(ctx, typedSecretName, &kSecret); err != nil {
		err = fmt.Errorf("failed to fetch secret %s: %w", typedSecretName, err)
		return
	}

	secUUIDBytes, ok := kSecret.Data[SecretUUIDDataKey]
	if !ok {
		err = fmt.Errorf("failed to get %s key from secret %s", SecretUUIDDataKey, namespace.Name)
		return
	}
	secUUID := string(secUUIDBytes)

	var secVersion *yav.GetVersionResponse
	secVersion, err = reader.GetYavClient().GetVersion(ctx, secUUID)

	if err != nil {
		err = fmt.Errorf("failed to fetch secret %s from YAV: %w", secUUID, err)
		return
	}
	if secVersion == nil {
		err = fmt.Errorf("YAV response for call GetVersion is nil: secret %s", secUUID)
		return
	}
	if secVersion.Status != "ok" {
		err = fmt.Errorf(
			"YAV response for call GetSecretVersionsByTokens is not OK: status %s code %s secret %s",
			secVersion.Status,
			secVersion.Code,
			secUUID,
		)
		return
	}

	for _, val := range secVersion.Version.Values {
		if val.Key == CommonTokenYavKey {
			token = val.Value
			return
		}
	}

	err = fmt.Errorf("secret not found in YAV secret %s", secUUID)
	return

}

func LoadProviderToken(
	reader SecretReader,
	ctx context.Context,
	namespace *corev1.Namespace,
	provider string,
	selfTvmID uint64,
) (token string, err error) {
	secretName := MakeK8sSecretName(provider)
	delegationSecretName := MakeK8sDelegationTokenSecretName(provider, selfTvmID)
	var delegationToken string
	var kSecret corev1.Secret

	typedSecretName := types.NamespacedName{Namespace: namespace.Name, Name: delegationSecretName}
	if err = reader.GetK8sClient().Get(ctx, typedSecretName, &kSecret); err == nil {
		if tokenBytes, ok := kSecret.Data[DelegationTokenDataKey]; ok {
			delegationToken = string(tokenBytes)
		}
	}

	typedSecretName = types.NamespacedName{Namespace: namespace.Name, Name: secretName}
	if err = reader.GetK8sClient().Get(ctx, typedSecretName, &kSecret); err != nil {
		err = fmt.Errorf("failed to fetch secret %s: %w", typedSecretName, err)
		return
	}

	secUUIDBytes, ok := kSecret.Data[SecretUUIDDataKey]
	if !ok {
		err = fmt.Errorf("failed to get %s key from secret %s", SecretUUIDDataKey, namespace.Name)
		return
	}
	secUUID := string(secUUIDBytes)

	if len(delegationToken) == 0 {
		tokenBytes, ok := kSecret.Data[DelegationTokenDataKey]
		if !ok {
			err = fmt.Errorf("failed to get %s key from secret %s", DelegationTokenDataKey, namespace.Name)
			return
		}
		delegationToken = string(tokenBytes)
	}

	srvTicket, err := reader.GetTvmClient().GetServiceTicketForAlias(ctx, "yav")
	if err != nil {
		err = fmt.Errorf("failed to get service ticket from TVM: %w", err)
		return
	}

	yavReq := yav.GetSecretVersionsByTokensRequest{
		TokenizedRequests: []yav.TokenizedRequest{
			{
				SecretUUID:    secUUID,
				Token:         delegationToken,
				Signature:     string(namespace.UID),
				ServiceTicket: srvTicket,
			},
		},
	}
	yavRsp, err := reader.GetYavClient().GetSecretVersionsByTokens(ctx, yavReq)
	if err != nil {
		err = fmt.Errorf("failed to fetch secret %s from YAV: %w", secUUID, err)
		return
	}
	if yavRsp == nil {
		err = fmt.Errorf("YAV response for call GetSecretVersionsByTokens is nil: secret %s", secUUID)
		return
	}
	if yavRsp.Status != "ok" {
		err = fmt.Errorf("YAV response for call GetSecretVersionsByTokens is not OK: status %s code %s secret %s", yavRsp.Status, yavRsp.Code, secUUID)
		return
	}
	for _, yavSecret := range yavRsp.Secrets {
		if yavSecret.Token != delegationToken {
			continue
		}
		for _, val := range yavSecret.Values {
			if val.Key == CommonTokenYavKey {
				token = val.Value
				return
			}
		}
	}
	err = fmt.Errorf("secret %s not found in YAV by delegation token", secUUID)
	return
}

// LoadToken TODO remove this method
func LoadToken(
	reader SecretReader,
	ctx context.Context,
	namespace *corev1.Namespace,
	secretName string,
	tokenLabel string,
	tokenKey string,
) (token string, err error) {
	secUUID, ok := namespace.Labels[tokenLabel]
	if !ok {
		err = fmt.Errorf("failed to find %s label key in namespace %s", tokenLabel, namespace.Name)
		return
	}

	var kSecret corev1.Secret
	if err = reader.GetK8sClient().Get(ctx, types.NamespacedName{Namespace: namespace.Name, Name: secretName}, &kSecret); err != nil {
		err = fmt.Errorf("failed to fetch secret %s: %w", namespace.Name, err)
		return
	}

	tokenBytes, ok := kSecret.Data[DelegationTokenDataKey]
	if !ok {
		err = fmt.Errorf("failed to get %s key from secret %s", DelegationTokenDataKey, namespace.Name)
		return
	}
	delegationToken := string(tokenBytes)

	srvTicket, err := reader.GetTvmClient().GetServiceTicketForAlias(ctx, "yav")
	if err != nil {
		err = fmt.Errorf("failed to get service ticket from TVM: %w", err)
		return
	}

	yavReq := yav.GetSecretVersionsByTokensRequest{
		TokenizedRequests: []yav.TokenizedRequest{
			{
				SecretUUID:    secUUID,
				Token:         delegationToken,
				Signature:     string(namespace.UID),
				ServiceTicket: srvTicket,
			},
		},
	}
	yavRsp, err := reader.GetYavClient().GetSecretVersionsByTokens(ctx, yavReq)
	if err != nil {
		err = fmt.Errorf("failed to fetch secret %s from YAV: %w", secUUID, err)
		return
	}
	if yavRsp == nil {
		err = fmt.Errorf("YAV response for call GetSecretVersionsByTokens is nil: secret %s", secUUID)
		return
	}
	if yavRsp.Status != "ok" {
		err = fmt.Errorf("YAV response for call GetSecretVersionsByTokens is not OK: status %s code %s secret %s", yavRsp.Status, yavRsp.Code, secUUID)
		return
	}
	for _, yavSecret := range yavRsp.Secrets {
		if yavSecret.Token != delegationToken {
			continue
		}
		for _, val := range yavSecret.Values {
			if val.Key == tokenKey {
				token = val.Value
				return
			}
		}
	}
	err = fmt.Errorf("secret %s not found in YAV by delegation token", secUUID)
	return
}
