package secrets

import (
	"context"
	"fmt"
	"log"
	"strings"

	corev1 "k8s.io/api/core/v1"
	v1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/types"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	"a.yandex-team.ru/infra/infractl/internal/environs"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/infra/infractl/util/kubeutil"
	"a.yandex-team.ru/infra/infractl/util/oauthutil"
	"a.yandex-team.ru/library/go/yandex/yav"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

const (
	YpTokenYavKey     = "infractl.yp_token"
	AwacsTokenYavKey  = "infractl.awacs_token"
	CommonTokenYavKey = "infractl.token"

	DelegationTokenDataKey = "delegation-token"
	SecretUUIDDataKey      = "secret-uuid"

	TokensSecretName          = "infractl-tokens"
	TokensSecretNameV2        = "infractl-tokens-v2"
	DelegationTokenSecretName = "infractl-delegation-token"

	// TokensSecretUUIDLabelKey Deprecated
	TokensSecretUUIDLabelKey = "k.yandex-team.ru/infractl.tokens.yav_secret_uuid"
	// YpTokenSecretUUIDLabelKey Deprecated
	YpTokenSecretUUIDLabelKey = "k.yandex-team.ru/infractl.yp_token.yav_secret_uuid"
)

type YavClient struct {
	*httpyav.Client
}

func MakeK8sSecretName(provider string) string {
	return fmt.Sprintf("%v-%v", TokensSecretName, provider)
}

func MakeK8sDelegationTokenSecretName(provider string, tvmID uint64) string {
	return fmt.Sprintf("%v-%v-%v", DelegationTokenSecretName, provider, tvmID)
}

func MakeYavSecretName(ns, provider string) string {
	return fmt.Sprintf("infractl-%s-token-%s", ns, provider)
}

func (c YavClient) CreateSecretWithRole(ctx context.Context, secretName string, req yav.SecretRoleRequest) (string, error) {
	createSecretResponse, err := c.CreateSecret(ctx, yav.SecretRequest{
		Name: secretName,
	})
	if err := yavErr(&createSecretResponse.Response, err); err != nil {
		return "", err
	}
	secretUUID := createSecretResponse.SecretUUID

	secretRoleResponse, err := c.AddSecretRole(ctx, secretUUID, req)
	if err := yavErr(secretRoleResponse, err); err != nil {
		return "", fmt.Errorf("cannot add secret role: %w", err)
	}
	return secretUUID, nil
}

func (c YavClient) CreateProviderSecret(ctx context.Context, ns, abcSlug, provider string) (string, error) {
	abcToken, err := oauthutil.GetToken(ctx, "ABC_TOKEN")
	if err != nil {
		return "", err
	}
	abcClient := abc.NewClient(abcToken)

	abcID, err := abcClient.GetServiceIDBySlug(abcSlug)
	if err != nil {
		return "", fmt.Errorf("cannot get service id from abc: %w", err)
	}

	return c.CreateSecretWithRole(ctx, MakeYavSecretName(ns, provider), yav.SecretRoleRequest{
		AbcID: uint(abcID),
		Role:  yav.RoleOwner,
	})
}

type FindSecretsOptions struct {
	Alias         string
	Author        string
	ABCSlugReader string
	Limit         uint
}

func (opts FindSecretsOptions) FindRole(roles []yav.SecretRole) *yav.SecretRole {
	for _, role := range roles {
		if role.AbcSlug != opts.ABCSlugReader {
			continue
		}
		if strings.ToUpper(role.RoleSlug) != yav.RoleReader {
			continue
		}
		return &role
	}
	return nil
}

func (c YavClient) FindSecrets(ctx context.Context, opts FindSecretsOptions) ([]string, error) {
	rsp, err := c.GetSecrets(ctx, yav.GetSecretsRequest{
		Query:     opts.Alias,
		QueryType: yav.SecretQueryTypeExact,
		// search among first `Limit` secrets only
		PageSize: opts.Limit,
	})
	if err = yavErr(&rsp.Response, err); err != nil {
		return nil, fmt.Errorf("failed to get secrets from yav: %w", err)
	}
	secrets := make([]string, 0, opts.Limit)
	for _, sec := range rsp.Secrets {
		// query may search not only by name but also by comment
		if sec.Name != opts.Alias {
			continue
		}
		if sec.CreatorLogin != opts.Author {
			continue
		}
		if opts.ABCSlugReader != "" {
			if opts.FindRole(sec.SecretRoles) == nil {
				continue
			}
		}
		secrets = append(secrets, sec.SecretUUID)
	}
	return secrets, nil
}

func yavErr(r *yav.Response, err error) error {
	if err != nil {
		return err
	}
	return r.Err()
}

func (c YavClient) SetSecretField(ctx context.Context, secretUUID, key, value string) error {
	versionResponse, err := c.GetVersion(ctx, secretUUID)
	var records []yav.Value
	if err = yavErr(&versionResponse.Response, err); err != nil {
		if err != yav.ErrNonExistentEntity {
			return err
		}
	} else {
		records = versionResponse.Version.Values
	}
	newVersionRequest := yav.CreateVersionRequest{}
	for _, record := range records {
		if record.Key != key {
			newVersionRequest.Values = append(newVersionRequest.Values, record)
		}
	}
	newVersionRequest.Values = append(newVersionRequest.Values, yav.Value{
		Key:   key,
		Value: value,
	})
	createVersionResponse, err := c.CreateVersion(ctx, secretUUID, newVersionRequest)
	if err := yavErr(&createVersionResponse.Response, err); err != nil {
		return fmt.Errorf("cannot create version: %w", err)
	}

	return nil
}

func (c YavClient) delegateNamespaceSecret(
	ctx context.Context,
	secretUUID, namespaceUUID string,
	tvmID uint64,
) (string, error) {
	response, err := c.CreateToken(ctx, secretUUID, yav.CreateTokenRequest{
		Signature:   namespaceUUID,
		TVMClientID: tvmID,
	})
	if err := yavErr(&response.Response, err); err != nil {
		return "", err
	}
	return response.Token, nil
}

func makeSecret(namespace, name string, fields ...string) *corev1.Secret {
	secret := &corev1.Secret{
		ObjectMeta: v1.ObjectMeta{
			Namespace: namespace,
			Name:      name,
		},
		StringData: map[string]string{},
		Type:       corev1.SecretTypeOpaque,
	}
	secret.APIVersion = corev1.SchemeGroupVersion.String()
	secret.Kind = "Secret"

	if len(fields)%2 != 0 {
		panic("fields ain't k=>v pairs, even number of fields arguments expected")
	}
	var k string
	for i, field := range fields {
		if i%2 == 0 {
			k = field
		} else {
			secret.StringData[k] = field
		}
	}

	return secret
}

type DelegateSecretOptions struct {
	KubeClient         *kubeutil.Client
	NamespaceObjectKey types.NamespacedName
	Provider           string
	SecretUUID         string
	TvmAliasesAndIDs   []string
}

func (c YavClient) DelegateSecret(ctx context.Context, opts DelegateSecretOptions) error {
	ns := &corev1.Namespace{}
	if err := opts.KubeClient.Get(ctx, opts.NamespaceObjectKey, ns); err != nil {
		return fmt.Errorf("failed to fetch created object from k8s: %w", err)
	}

	env := environs.GetCurrentEnviron()
	tvmIDs, err := env.ResolveTvmIDs(opts.TvmAliasesAndIDs)
	if err != nil {
		return err
	}

	var secretName string
	var secret *corev1.Secret
	for _, tvmID := range tvmIDs {
		delegationToken, err := c.delegateNamespaceSecret(
			ctx,
			opts.SecretUUID,
			string(ns.ObjectMeta.UID),
			tvmID,
		)
		if err != nil {
			return fmt.Errorf("token delegation failed: %w", err)
		}

		secretName = MakeK8sDelegationTokenSecretName(opts.Provider, tvmID)
		secret = makeSecret(ns.Name, secretName, DelegationTokenDataKey, delegationToken)

		if _, err := opts.KubeClient.PutObject(ctx, client.ObjectKeyFromObject(secret), secret); err != nil {
			return fmt.Errorf("failed to push secret to k8s: %w", err)
		}
	}

	return nil
}

func MustCreateClient(token string) YavClient {
	yavClient, err := httpyav.NewClient(httpyav.WithOAuthToken(token))
	if err != nil {
		log.Fatal(err)
	}
	return YavClient{yavClient}
}

type EnsureSecretOptions struct {
	KubeClient       *kubeutil.Client
	YAVClient        *YavClient
	Provider         string
	Namespace        *corev1.Namespace
	TVMAliasesAndIDs []string // TODO(reddi): it must be set with NeedsDelegation option in Provider
}

func EnsureSecretForNamespace(ctx context.Context, opts EnsureSecretOptions) (string, error) {
	var secretUUID string
	curSecret, err := GetKubeSecret(ctx, opts.KubeClient, opts.Provider, opts.Namespace)
	if err != nil {
		return "", fmt.Errorf("failed to check if secret for %s-token exists in k8s", opts.Provider)
	}
	if curSecret != nil {
		if secretUUIDBytes, ok := curSecret.Data[SecretUUIDDataKey]; ok {
			secretUUID = string(secretUUIDBytes)
		}
	}
	if len(secretUUID) == 0 {
		secretUUID, err = opts.YAVClient.CreateProviderSecret(ctx, opts.Namespace.Name, opts.Namespace.Labels[labels.ABC], opts.Provider)
		if err != nil {
			return "", fmt.Errorf("namespace secret creation failed: %w", err)
		}
	}

	secretName := MakeK8sSecretName(opts.Provider)
	secret := makeSecret(opts.Namespace.Name, secretName, SecretUUIDDataKey, secretUUID)
	if _, err := opts.KubeClient.PutObject(ctx, client.ObjectKeyFromObject(secret), secret); err != nil {
		return "", fmt.Errorf("failed to push secret to k8s: %w", err)
	}

	prov := GetProvider(opts.Provider)
	if prov == nil {
		return "", fmt.Errorf("provider %q not fount", opts.Provider)
	}
	if prov.NeedsDelegation {
		err = opts.YAVClient.DelegateSecret(ctx, DelegateSecretOptions{
			KubeClient:         opts.KubeClient,
			NamespaceObjectKey: client.ObjectKeyFromObject(opts.Namespace),
			Provider:           opts.Provider,
			SecretUUID:         secretUUID,
			TvmAliasesAndIDs:   opts.TVMAliasesAndIDs,
		})
		if err != nil {
			return "", fmt.Errorf("failed to delegate secret %q to namespace %q: %w", secretUUID, opts.Namespace.Name, err)
		}
	}

	return secretUUID, nil
}

func GetKubeSecret(ctx context.Context, kubeClient *kubeutil.Client, provider string, ns *corev1.Namespace) (*corev1.Secret, error) {
	secretName := MakeK8sSecretName(provider)
	secret := &corev1.Secret{}
	err := kubeClient.Get(ctx, client.ObjectKey{Namespace: ns.Name, Name: secretName}, secret)
	if err != nil {
		if client.IgnoreNotFound(err) == nil {
			return nil, nil
		}
		return nil, err
	}
	return secret, nil
}
