package validation

import (
	"context"
	"fmt"

	"github.com/go-logr/logr"
	corev1 "k8s.io/api/core/v1"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	dpv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/v1"
	dsv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/deploy/validation"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

func ValidateIsUserABCMember(abcClient *abc.Client, log logr.Logger, abcSlug, username string) (bool, string, error) {
	isMember, err := abcClient.IsUserServiceMember(abcSlug, username)
	if err != nil {
		log.Error(err, "failed to list members for ABC service", "abc", abcSlug)
		return false, "", fmt.Errorf("failed to check if user %q is a member of ABC service %q: %w", username, abcSlug, err)
	}
	if isMember {
		return true, "", nil
	}
	log.Info("validation failed: user does not belong to ABC service", "user", username, "abc", abcSlug)
	// TODO(reddi): make idm url for asking role by ABC slug
	msg := fmt.Sprintf(
		"User %q does not belong to ABC service %q. Please specify ABC service the user belong to "+
			"or ask corresponding user role in provided ABC service: https://abc.yandex-team.ru/services/%v/", username, abcSlug, abcSlug)
	return false, msg, nil
}

func ValidateNamespace(abcClient *abc.Client, log logr.Logger, ns *corev1.Namespace, username string) (bool, string, error) {
	abcSlug, exists := ns.Labels[labels.ABC]
	if !exists {
		log.Info("validation failed: ABC service slug is not found in labels", "namespace", ns.Name)
		msg := fmt.Sprintf(
			"ABC service slug is required, but not found in namespace labels. "+
				"Please set your ABC service slug in %q labels key", labels.ABC,
		)
		return false, msg, nil
	}
	return ValidateIsUserABCMember(abcClient, log, abcSlug, username)
}

func ValidateProject(
	ctx context.Context,
	log logr.Logger,
	deployClient *dclient.DeployClient,
	abcClient *abc.Client,
	abcSlug string,
	curKProject, newKProject *dpv1.DeployProject,
) (bool, string, error) {
	if curKProject == nil {
		name := newKProject.GetName()
		nsName := newKProject.GetNamespace()
		if name != nsName {
			return false, fmt.Sprintf("Deploy project name %q is not equal to namespace name %q", name, nsName), nil
		}
	}
	return validation.ValidateDeployObject(ctx, log, deployClient, abcClient, abcSlug, curKProject, newKProject)
}

func validateStageProjectID(ctx context.Context, deployClient *dclient.DeployClient, kStage *dsv1.DeployStage) (bool, string, error) {
	dStage, err := deployClient.Fetch(ctx, 0, kStage)
	if err != nil {
		return false, "", fmt.Errorf("failed to get deploy stage %s/%s: %w", kStage.GetNamespace(), kStage.GetName(), err)
	}
	// Stage in YP does not exist
	if dStage == nil {
		return true, "", nil
	}
	ypStage, ok := dStage.GetYPObject().(*ypapi.TStage)
	if !ok {
		return false, "", fmt.Errorf("failed to cast deploy stage to YP stage")
	}
	meta := ypStage.GetMeta()
	if meta.GetProjectId() != kStage.Namespace {
		return false, fmt.Sprintf("Namespace name %q does not match YP project_id %q", kStage.Namespace, meta.GetProjectId()), nil
	}

	return true, "", nil
}

func ValidateStage(
	ctx context.Context,
	log logr.Logger,
	deployClient *dclient.DeployClient,
	abcClient *abc.Client,
	abcSlug string,
	curKStage *dsv1.DeployStage,
	newKStage *dsv1.DeployStage,
) (bool, string, error) {
	if curKStage == nil {
		log.Info("object is being just created, need to check project_id")
		isValid, msg, err := validateStageProjectID(ctx, deployClient, newKStage)
		if err != nil {
			log.Error(err, "namespace mismatch")
			return isValid, msg, err
		}
	}
	return validation.ValidateDeployObject(ctx, log, deployClient, abcClient, abcSlug, curKStage, newKStage)
}
