package client

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/infra/infractl/services/oauth-to-login/types"
)

type endpoint = string

const prestable = endpoint("https://oauth.dev-k.yandex-team.ru")
const production = endpoint("https://oauth.k.yandex-team.ru")

type Client struct {
	client *resty.Client
}

func NewClient(remote endpoint) *Client {
	r := resty.
		New().
		SetBaseURL(remote).
		SetRedirectPolicy(resty.NoRedirectPolicy()).
		SetHeader("Accept", "application/json").
		SetHeader("User-Agent", "infra/infractl/services/oauth-to-login/client").
		SetTimeout(2000 * time.Millisecond)

	return &Client{client: r}
}

func NewPrestableClient() *Client {
	return NewClient(prestable)
}

func NewProductionClient() *Client {
	return NewClient(production)
}

func (c Client) GetInfo(ctx context.Context, token string) (*types.LoginInfo, error) {
	httpReq := c.client.R().
		SetContext(ctx).
		SetAuthToken(token)
	rsp, err := httpReq.Get(types.LoginInfoPath)
	if err != nil {
		return nil, err
	}

	if rsp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("request failed with: %v", rsp.Body())
	}

	var result types.LoginInfo
	if err = json.Unmarshal(rsp.Body(), &result); err != nil {
		return nil, fmt.Errorf("failed to parse response: %w", err)
	}
	return &result, nil
}
