package main

import (
	"encoding/json"
	"fmt"
	"net"
	"net/http"
	"os"
	"strconv"
	"strings"

	"github.com/go-logr/logr"
	"github.com/spf13/pflag"

	"a.yandex-team.ru/infra/infractl/clients/blackbox"
	"a.yandex-team.ru/infra/infractl/internal/log"
	"a.yandex-team.ru/infra/infractl/services/oauth-to-login/types"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
)

type handler struct {
	log            logr.Logger
	blackboxClient blackbox.LoginFromOauthGetter
}

func (h *handler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	log := h.log.WithValues("remote-addr", r.RemoteAddr)
	defer log.Info("Request processed")
	accept := r.Header.Get("Accept")
	if len(accept) == 0 || !(strings.HasPrefix(accept, "application/json") || accept == "*/*" || accept == "*") {
		log.Info("invalid accept header", "Accept", accept)
		http.Error(w, ErrorNonJSON.Error(), http.StatusNotAcceptable)
		return
	}

	authorization := r.Header.Get("Authorization")
	if len(authorization) == 0 {
		log.Info("No authorization header")
		http.Error(w, ErrorNoAuthorization.Error(), http.StatusUnauthorized)
		return
	}

	method, token, ok := strings.Cut(authorization, " ")
	if !ok || !(method == "OAuth" || method == "Bearer") {
		log.Info("Invalid authorization method", "method-4-chars", method[:4])
		http.Error(w, ErrorInvalidMethod.Error(), http.StatusUnauthorized)
		return
	}

	var userIP string
	if forwardedFor := r.Header.Get("X-Forwarded-For"); len(forwardedFor) > 0 {
		parts := strings.Split(forwardedFor, ", ")
		userIP = parts[0]
	} else {
		ip, _, err := net.SplitHostPort(r.RemoteAddr)
		if err != nil {
			log.Error(err, "Unparsable remote addr")
			http.Error(w, err.Error(), http.StatusBadRequest)
			return
		}
		userIP = ip
	}

	login, uid, err := h.blackboxClient.GetLoginFromOauth(r.Context(), token, userIP)
	if err != nil {
		log.Error(err, "Cannot get user info")
		http.Error(w, ErrorUserInfoFailed(err).Error(), http.StatusUnauthorized)
		return
	}

	b, err := json.Marshal(types.LoginInfo{Login: login, UID: uid})
	if err != nil {
		log.Error(err, "Cannot marshal response")
		http.Error(w, fmt.Sprintf("Failed to marshal json: %v", err), http.StatusInternalServerError)
		return
	}

	w.Header().Set("Content-Type", "application/json")
	w.Header().Set("Content-Length", strconv.Itoa(len(b)))
	_, err = w.Write(b)
	if err != nil {
		log.Error(err, "Cannot write response")
	}
}

func main() {
	tvmID := pflag.Uint64("tvm-id", 0, "Own TVM ID")
	port := pflag.UintP("port", "p", 8080, "Port to bind to")

	opts := log.Bind(pflag.CommandLine)

	pflag.Parse()

	svcLog := opts.ConfigureLogger()

	if *port >= 65536 {
		svcLog.Info("Invalid port value, must be < 65536", "value", *port)
		os.Exit(1)
	}

	tvmSecret := os.Getenv("TVM_SECRET")
	if tvmSecret == "" {
		svcLog.Info("unable to get TVM_SECRET env variable")
		os.Exit(1)
	}
	tvmSettings := tvmauth.TvmAPISettings{
		SelfID: tvm.ClientID(*tvmID),
		ServiceTicketOptions: tvmauth.NewAliasesOptions(
			tvmSecret,
			map[string]tvm.ClientID{
				"blackbox": httpbb.IntranetEnvironment.TvmID,
			},
		),
	}

	tvmc, err := tvmauth.NewAPIClient(tvmSettings, &nop.Logger{})
	if err != nil {
		svcLog.Error(err, "unable to create tvm client")
		os.Exit(1)
	}

	bbClient, err := blackbox.MakeClient(tvmc)
	if err != nil {
		svcLog.Error(err, "failed to create blackbox client")
		os.Exit(1)
	}

	http.Handle(types.LoginInfoPath, &handler{log: svcLog, blackboxClient: bbClient})
	http.HandleFunc("/ping", func(w http.ResponseWriter, r *http.Request) {
		_, _ = w.Write([]byte("Ok"))
	})
	err = http.ListenAndServe(fmt.Sprintf(":%v", *port), nil)
	if err != nil {
		svcLog.Error(err, "Execution finished")
	} else {
		svcLog.Info("Execution finished")
	}
}
