package deployutil

import (
	"fmt"
	"regexp"
	"sort"

	"a.yandex-team.ru/yp/go/proto/podagent"
	"a.yandex-team.ru/yp/go/proto/ypapi"
)

var (
	secretRe = regexp.MustCompile(`^\$\{([a-z0-9_-]+):([a-z0-9_-]+):([^}]+)\}$`)
)

type Secret struct {
	ID      string
	Version string
	Key     string
}

func (s *Secret) MakeAlias() string {
	return fmt.Sprintf("%s:%s", s.ID, s.Version)
}

func (s *Secret) MakeSecretRef() *ypapi.TSecretRef {
	return &ypapi.TSecretRef{
		SecretId:      &s.ID,
		SecretVersion: &s.Version,
	}
}

func ParseSecret(value string) (*Secret, bool) {
	match := secretRe.FindStringSubmatch(value)
	if match == nil {
		return nil, false
	}

	return &Secret{
		ID:      match[1],
		Version: match[2],
		Key:     match[3],
	}, true
}

func ParseEnvValues(rawEnvVars map[string]string, sortByKey bool) ([]*podagent.TEnvVar, map[string]*ypapi.TSecretRef) {
	envVars := make([]*podagent.TEnvVar, 0, len(rawEnvVars))

	refs := map[string]*ypapi.TSecretRef{}
	for key, value := range rawEnvVars {
		secret, ok := ParseSecret(value)
		var envValue *podagent.TEnvVarValue
		if ok {
			alias := secret.MakeAlias()
			envValue = &podagent.TEnvVarValue{
				Value: &podagent.TEnvVarValue_SecretEnv{
					SecretEnv: &podagent.SecretSelector{
						Alias: alias,
						Id:    secret.Key,
					},
				},
			}
			refs[alias] = secret.MakeSecretRef()
		} else {
			envValue = &podagent.TEnvVarValue{
				Value: &podagent.TEnvVarValue_LiteralEnv{
					LiteralEnv: &podagent.LiteralEnvSelector{
						Value: value,
					},
				},
			}
		}

		envVars = append(envVars, &podagent.TEnvVar{
			Name:  key,
			Value: envValue,
		})
	}

	if sortByKey {
		sort.Slice(envVars, func(i, j int) bool {
			return envVars[i].Name < envVars[j].Name
		})
	}

	return envVars, refs
}
