package kubeutil

import (
	"bytes"
	"context"
	"encoding/pem"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"reflect"
	"strings"

	"k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/runtime/schema"

	"k8s.io/client-go/kubernetes/scheme"
	clientcmdapi "k8s.io/client-go/tools/clientcmd/api"
	"sigs.k8s.io/controller-runtime/pkg/client"

	"a.yandex-team.ru/infra/infractl/internal/environs"
	"a.yandex-team.ru/infra/infractl/internal/kubeconfig"
	"a.yandex-team.ru/infra/infractl/util/oauthutil"
	"a.yandex-team.ru/library/go/certifi"
)

const passportAuthURL = "https://passport.yandex-team.ru/auth"

type authErrorChecker struct {
	rt http.RoundTripper
}

func webauthAuthenticatedError(rsp *http.Response) bool {
	if rsp.StatusCode != 302 {
		return false
	}
	loc := rsp.Header.Get("Location")
	return strings.HasPrefix(loc, passportAuthURL)
}

func webauthAuthorizedError(statusCode int, body []byte) bool {
	if statusCode != 403 {
		return false
	}
	return bytes.Equal(body, []byte("You are not authorized to access this resource."))
}

func (c *authErrorChecker) RoundTrip(req *http.Request) (*http.Response, error) {
	rsp, err := c.rt.RoundTrip(req)
	if err != nil {
		return rsp, err
	}
	if rsp.StatusCode != 302 && rsp.StatusCode != 403 {
		return rsp, err
	}
	if webauthAuthenticatedError(rsp) {
		return nil, fmt.Errorf("you are not authenticated in webauth, please check user token specified in `~/.kube/config`")
	}
	body, readErr := io.ReadAll(rsp.Body)
	defer func() {
		_ = rsp.Body.Close()
		rsp.Body = ioutil.NopCloser(bytes.NewBuffer(body))
	}()
	if readErr != nil {
		return rsp, err
	}
	if webauthAuthorizedError(rsp.StatusCode, body) {
		return nil, fmt.Errorf("you are not authorized in webauth, please request access to infractl: https://docs.yandex-team.ru/infractl/howto#access")
	}
	return rsp, err
}

type Client struct {
	client.Client
}

func MakeClient() Client {
	config, err := kubeconfig.GetKubeClientConfig().ClientConfig()
	if err != nil {
		log.Fatalf("Failed to read kube config: %v", err)
	}

	wt := config.WrapTransport
	config.WrapTransport = func(rt http.RoundTripper) http.RoundTripper {
		if wt != nil {
			rt = wt(rt)
		}
		return &authErrorChecker{
			rt: rt,
		}
	}

	c, err := client.New(config, client.Options{
		Scheme: scheme.Scheme,
		Opts:   client.WarningHandlerOptions{},
	})
	if err != nil {
		log.Fatalf("Failed to create k8s client: %v", err)
	}
	return Client{Client: c}
}

func MakeInfraCtlKubeConfig(ctx context.Context, kubeContexts []string) *clientcmdapi.Config {
	cfg := clientcmdapi.NewConfig()
	cfg.APIVersion = clientcmdapi.SchemeGroupVersion.String()
	cfg.Kind = "Config"

	var caData []byte
	for _, cert := range certifi.InternalCAs() {
		caData = append(caData, pem.EncodeToMemory(&pem.Block{
			Type:  "CERTIFICATE",
			Bytes: cert.Raw,
		})...)
	}

	for _, kubeContext := range kubeContexts {
		e := environs.Environs[kubeContext]
		if e == nil {
			continue
		}
		token, err := oauthutil.GetTokenForEnviron(ctx, e, oauthutil.InfractlTokenEnvVarName)
		if err != nil {
			log.Fatalf("Failed to obtain k8s OAuth token: %s", err)
		}
		opts := e.ConfigOpts
		cfg.Clusters[opts.Cluster] = &clientcmdapi.Cluster{Server: opts.Server, CertificateAuthorityData: caData}
		cfg.AuthInfos[opts.User] = &clientcmdapi.AuthInfo{Token: token}
		cfg.Contexts[opts.Context] = &clientcmdapi.Context{Cluster: opts.Cluster, AuthInfo: opts.User}
	}
	return cfg
}

func (c *Client) PutObject(ctx context.Context, key client.ObjectKey, spec client.Object) (bool, error) {
	valType := reflect.ValueOf(spec)
	if valType.Kind() == reflect.Ptr {
		valType = valType.Elem()
	}
	obj := reflect.New(valType.Type()).Interface().(client.Object)

	err := c.Get(ctx, key, obj)
	if err != nil {
		if errors.IsNotFound(err) {
			if createErr := c.Create(ctx, spec); createErr != nil {
				return false, createErr
			}
			return true, nil
		}
		return false, err
	}

	// FIXME (torkve) is there a better way to save all metadata and update anything else, but with generic object?
	spec.SetResourceVersion(obj.GetResourceVersion())
	if obj.GetAnnotations() == nil {
		obj.SetAnnotations(map[string]string{})
	}
	annotations := obj.GetAnnotations()
	for k, v := range spec.GetAnnotations() {
		annotations[k] = v
	}
	spec.SetAnnotations(annotations)
	if err = c.Update(ctx, spec); err != nil {
		return false, err
	}
	return obj.GetGeneration() != spec.GetGeneration(), nil
}

func (c *Client) ResolveKind(kind string) (schema.GroupVersionKind, error) {
	mapper := c.RESTMapper()

	kind, err := mapper.ResourceSingularizer(kind)
	if err != nil {
		return schema.GroupVersionKind{}, err
	}
	gvk, err := mapper.KindFor(schema.GroupVersionResource{Resource: kind})
	if err != nil {
		return schema.GroupVersionKind{}, err
	}
	return gvk, nil
}
