package deployproject

import (
	"context"
	"fmt"
	"net/http"

	admissionv1 "k8s.io/api/admission/v1"
	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/types"
	"sigs.k8s.io/controller-runtime/pkg/client"
	logf "sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/webhook/admission"

	"a.yandex-team.ru/infra/infractl/clients/abc"
	dpv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/project/v1"
	dsv1 "a.yandex-team.ru/infra/infractl/controllers/deploy/api/stage/v1"
	dclient "a.yandex-team.ru/infra/infractl/internal/deploy/client"
	"a.yandex-team.ru/infra/infractl/internal/labels"
	"a.yandex-team.ru/infra/infractl/internal/validation"
)

var whlog = logf.Log.WithName("dp-webhook")

// DeployProjectValidator validates Deployprojects
type DeployProjectValidator struct {
	client       client.Client
	deployClient *dclient.DeployClient
	abcClient    *abc.Client
	decoder      *admission.Decoder
}

func NewDeployProjectValidator(client client.Client, deployClient *dclient.DeployClient, abcClient *abc.Client) *DeployProjectValidator {
	return &DeployProjectValidator{client: client, deployClient: deployClient, abcClient: abcClient}
}

func (v *DeployProjectValidator) Handle(ctx context.Context, req admission.Request) admission.Response {
	if req.Operation == admissionv1.Delete {
		dsl := &dsv1.DeployStageList{}
		if err := v.client.List(ctx, dsl, client.InNamespace(req.Namespace)); err != nil {
			return admission.Errored(http.StatusBadRequest, fmt.Errorf("failed to fetch deploy stages for project: %w", err))
		}
		if len(dsl.Items) != 0 {
			return admission.Denied("deploy project has undeleted deploy stages")
		}
		return admission.Allowed("")
	}

	kProject := &dpv1.DeployProject{}
	if err := v.decoder.Decode(req, kProject); err != nil {
		return admission.Errored(http.StatusBadRequest, err)
	}

	log := whlog.WithValues("namespace", kProject.Namespace, "name", kProject.Name)

	log.V(1).Info("validating DeployProject")

	curKProject := &dpv1.DeployProject{}
	err := v.client.Get(ctx, types.NamespacedName{Namespace: kProject.GetNamespace(), Name: kProject.GetName()}, curKProject)
	if client.IgnoreNotFound(err) == nil {
		curKProject = nil
	} else if err != nil {
		return admission.Errored(http.StatusBadRequest, fmt.Errorf("failed to get deploy project: %w", err))
	}

	ns := &corev1.Namespace{}
	if err = v.client.Get(ctx, types.NamespacedName{Name: kProject.GetName()}, ns); err != nil {
		return admission.Errored(http.StatusBadRequest, fmt.Errorf("failed to get namespace %q: %w", kProject.GetName(), err))
	}
	abcSlug := ns.Labels[labels.ABC]

	isValid, msg, err := validation.ValidateProject(ctx, log, v.deployClient, v.abcClient, abcSlug, curKProject, kProject)
	if err != nil {
		return admission.Errored(http.StatusBadRequest, fmt.Errorf("failed to validate deploy project: %w", err))
	}
	if !isValid {
		return admission.Denied(msg)
	}
	return admission.Allowed("")
}

func (v *DeployProjectValidator) InjectDecoder(d *admission.Decoder) error {
	v.decoder = d
	return nil
}
